# GitHub Secrets Setup Guide

## 📋 Required Secrets for Each Environment

### Staging Environment Secrets
Go to: https://github.com/aloahmilton/hirecameroon/settings/secrets/actions
Create these secrets under the "staging" environment:

```
STAGING_HOST=your-staging-server-ip-or-domain
STAGING_USER=your-ssh-username
STAGING_SSH_KEY=-----BEGIN OPENSSH PRIVATE KEY-----
your-private-ssh-key-here
-----END OPENSSH PRIVATE KEY-----
```

### Production Environment Secrets
Create these secrets under the "production" environment:

```
PROD_HOST=your-production-server-ip-or-domain
PROD_USER=your-ssh-username
PROD_SSH_KEY=-----BEGIN OPENSSH PRIVATE KEY-----
your-private-ssh-key-here
-----END OPENSSH PRIVATE KEY-----
PROD_DB_HOST=your-database-host
PROD_DB_USER=your-database-username
PROD_DB_PASSWORD=your-database-password
PROD_DB_NAME=hirecameroon_prod
```

### Repository-Level Secrets (for all environments)
```
AWS_S3_BACKUP_BUCKET=your-s3-backup-bucket-name
CLOUDFRONT_DISTRIBUTION_ID=your-cloudfront-distribution-id
DOCKER_USERNAME=your-dockerhub-username
DOCKER_PASSWORD=your-dockerhub-password
```

## 🔑 How to Generate SSH Keys

### Step 1: Generate SSH Key Pair
```bash
# Run this on your local machine
ssh-keygen -t rsa -b 4096 -C "deploy@hirecameroon.com" -f ~/.ssh/hirecameroon_deploy
```

### Step 2: Copy Public Key to Servers
```bash
# For staging server
ssh-copy-id -i ~/.ssh/hirecameroon_deploy.pub user@staging-server

# For production server
ssh-copy-id -i ~/.ssh/hirecameroon_deploy.pub user@production-server
```

### Step 3: Get Private Key Content
```bash
# Copy the entire private key content
cat ~/.ssh/hirecameroon_deploy
```

### Step 4: Add to GitHub Secrets
1. Go to repository Settings → Secrets and variables → Actions
2. Click "New repository secret"
3. Name: `STAGING_SSH_KEY` (for staging) or `PROD_SSH_KEY` (for production)
4. Value: Paste the entire private key content (including BEGIN and END lines)

## 🗄️ Database Setup

### Create Production Database
```sql
-- Run this on your production MySQL server
CREATE DATABASE hirecameroon_prod CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
CREATE USER 'hirecameroon_prod'@'localhost' IDENTIFIED BY 'your-secure-password';
GRANT ALL PRIVILEGES ON hirecameroon_prod.* TO 'hirecameroon_prod'@'localhost';
FLUSH PRIVILEGES;
```

### Create Staging Database
```sql
-- Run this on your staging MySQL server
CREATE DATABASE hirecameroon_staging CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
CREATE USER 'hirecameroon_staging'@'localhost' IDENTIFIED BY 'your-secure-password';
GRANT ALL PRIVILEGES ON hirecameroon_staging.* TO 'hirecameroon_staging'@'localhost';
FLUSH PRIVILEGES;
```

## 🔐 Environment Files Setup

### .env.staging (on staging server)
```env
APP_NAME=HireCameroon
APP_ENV=staging
APP_KEY=base64:your-generated-app-key
APP_DEBUG=false
APP_URL=https://staging.hirecameroon.com

DB_CONNECTION=mysql
DB_HOST=localhost
DB_PORT=3306
DB_DATABASE=hirecameroon_staging
DB_USERNAME=hirecameroon_staging
DB_PASSWORD=your-staging-db-password

# Add other configuration as needed
```

### .env.production (on production server)
```env
APP_NAME=HireCameroon
APP_ENV=production
APP_KEY=base64:your-generated-app-key
APP_DEBUG=false
APP_URL=https://hirecameroon.com

DB_CONNECTION=mysql
DB_HOST=localhost
DB_PORT=3306
DB_DATABASE=hirecameroon_prod
DB_USERNAME=hirecameroon_prod
DB_PASSWORD=your-production-db-password

# Add other configuration as needed
```

## 🧪 Testing Secrets

### Test SSH Connection
```bash
# Test staging connection
ssh -i ~/.ssh/hirecameroon_deploy user@staging-server "echo 'SSH works!'"

# Test production connection
ssh -i ~/.ssh/hirecameroon_deploy user@production-server "echo 'SSH works!'"
```

### Test Database Connection
```bash
# Test staging database
mysql -h localhost -u hirecameroon_staging -p hirecameroon_staging -e "SELECT 1;"

# Test production database
mysql -h localhost -u hirecameroon_prod -p hirecameroon_prod -e "SELECT 1;"
```

## 🚀 Deployment Testing

### Test Staging Deployment
1. Push to `staging` branch
2. Check GitHub Actions tab for deployment status
3. Visit https://staging.hirecameroon.com
4. Verify application loads and database is populated

### Test Production Deployment
1. Create a version tag: `git tag v1.0.0 && git push origin v1.0.0`
2. Check GitHub Actions for production deployment
3. Visit https://hirecameroon.com
4. Verify everything works correctly

## 🔧 Troubleshooting

### Common Issues:

1. **SSH Connection Failed**
   - Check if SSH key is correctly formatted (no extra spaces)
   - Verify server firewall allows SSH (port 22)
   - Ensure SSH key has correct permissions: `chmod 600 ~/.ssh/hirecameroon_deploy`

2. **Database Connection Failed**
   - Verify database credentials are correct
   - Check if database user has proper permissions
   - Ensure MySQL server is running and accessible

3. **Deployment Failed**
   - Check GitHub Actions logs for specific error messages
   - Verify all required secrets are set
   - Ensure server has all required software installed

4. **SSL Certificate Issues**
   - Run `sudo certbot --nginx -d yourdomain.com` on server
   - Check if domain DNS points to correct server IP

## 📞 Support

If you encounter issues:
1. Check GitHub Actions logs for detailed error messages
2. Verify all secrets are correctly set
3. Test SSH and database connections manually
4. Check server logs: `/var/log/nginx/error.log` and Laravel logs

---

**Remember**: Never commit sensitive information to the repository. Always use GitHub Secrets for sensitive data!