<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\Ad;
use App\Services\AdService;
use Illuminate\Support\Facades\Log;

class AdController extends Controller
{
    protected $adService;

    public function __construct(AdService $adService)
    {
        $this->middleware(['auth','verified'])->except(['recordImpression', 'recordClick']);
        $this->adService = $adService;
    }

    /**
     * Show ad creation form
     */
    public function create(Request $request)
    {
        $plans = DB::table('ad_plans')->orderBy('priority_weight', 'desc')->get();
        $selectedPlan = $request->query('plan');
        return view('ads.create', ['plans' => $plans, 'selectedPlan' => $selectedPlan]);
    }

    /**
     * List ads for the authenticated user
     */
    public function index(Request $request)
    {
        $user = $request->user();
        $filter = $request->query('filter', 'all');

        // Start building the query
        $query = Ad::query()
            ->where('user_id', $user->id)
            ->with('plan'); // Eager load plan relationship

        // Apply status filter
        if ($filter === 'active') {
            $query->where('status', 'active');
        } elseif ($filter === 'pending') {
            $query->whereIn('status', ['pending', 'pending_payment']);
        } elseif ($filter === 'expired') {
            $query->where('status', 'expired');
        }

        $ads = $query->orderByDesc('created_at')->get();

        // Get metrics for each ad
        $ads->each(function ($ad) {
            $ad->metrics = $this->adService->getAdStats($ad);
        });

        // Calculate totals
        $totalImpressions = $ads->sum(function ($ad) {
            return $ad->metrics['total_impressions'] ?? 0;
        });
        $totalClicks = $ads->sum(function ($ad) {
            return $ad->metrics['total_clicks'] ?? 0;
        });
        $avgCtr = $totalImpressions > 0 ? ($totalClicks / $totalImpressions) * 100 : 0;

        return view('ads.index', [
            'ads' => $ads,
            'filter' => $filter,
            'totalImpressions' => $totalImpressions,
            'totalClicks' => $totalClicks,
            'avgCtr' => $avgCtr,
        ]);
    }

    /**
     * Show a single ad (authorized)
     */
    public function show(Request $request, $adId)
    {
        $user = $request->user();
        $ad = DB::table('ads')->where('id', $adId)->first();
        if (!$ad) {
            abort(404);
        }
        if ($ad->user_id !== $user->id && !$user->is_admin) {
            abort(403);
        }

        $transactions = DB::table('ad_transactions')->where('ad_id', $ad->id)->orderByDesc('created_at')->get();

        return view('ads.show', ['ad' => $ad, 'transactions' => $transactions]);
    }

    /**
     * Record an ad impression
     */
    public function recordImpression(Request $request, Ad $ad)
    {
        try {
            if ($this->adService->recordImpression($ad, $request->route()?->getName())) {
                return response()->json(['success' => true]);
            }
            return response()->json(['success' => false, 'message' => 'Duplicate impression'], 200);
        } catch (\Exception $e) {
            Log::error('Failed to record ad impression', [
                'ad_id' => $ad->id,
                'error' => $e->getMessage()
            ]);
            return response()->json(['success' => false], 500);
        }
    }

    /**
     * Record an ad click and redirect to target URL
     */
    public function recordClick(Request $request, Ad $ad)
    {
        try {
            $this->adService->recordClick($ad);
            
            if ($request->wantsJson()) {
                return response()->json(['success' => true]);
            }

            return redirect()->away($ad->target_url);
        } catch (\Exception $e) {
            Log::error('Failed to record ad click', [
                'ad_id' => $ad->id,
                'error' => $e->getMessage()
            ]);
            
            if ($request->wantsJson()) {
                return response()->json(['success' => false], 500);
            }

            return redirect()->away($ad->target_url);
        }
    }

    /**
     * Get ad metrics
     */
    public function getMetrics(Request $request, Ad $ad)
    {
        if ($ad->user_id !== $request->user()->id && !$request->user()->is_admin) {
            abort(403);
        }

        return response()->json($this->adService->getAdStats($ad));
    }
}
