<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\EmailTemplate;
use Illuminate\Http\Request;

class EmailTemplateController extends Controller
{
    public function __construct()
    {
        $this->middleware('admin');
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $emailTemplates = EmailTemplate::orderBy('created_at', 'desc')->paginate(20);

        return view('admin.email-templates.index', compact('emailTemplates'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.email-templates.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:email_templates',
            'subject' => 'required|string|max:255',
            'content' => 'required|string',
            'variables' => 'nullable|array',
            'variables.*' => 'string',
            'type' => 'required|in:welcome,notification,marketing,system',
            'is_active' => 'boolean',
        ]);

        EmailTemplate::create($request->all());

        return redirect()->route('admin.email-templates.index')
            ->with('success', 'Email template created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(EmailTemplate $emailTemplate)
    {
        return view('admin.email-templates.show', compact('emailTemplate'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(EmailTemplate $emailTemplate)
    {
        return view('admin.email-templates.edit', compact('emailTemplate'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, EmailTemplate $emailTemplate)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:email_templates,name,' . $emailTemplate->id,
            'subject' => 'required|string|max:255',
            'content' => 'required|string',
            'variables' => 'nullable|array',
            'variables.*' => 'string',
            'type' => 'required|in:welcome,notification,marketing,system',
            'is_active' => 'boolean',
        ]);

        $emailTemplate->update($request->all());

        return redirect()->route('admin.email-templates.index')
            ->with('success', 'Email template updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(EmailTemplate $emailTemplate)
    {
        $emailTemplate->delete();

        return redirect()->route('admin.email-templates.index')
            ->with('success', 'Email template deleted successfully.');
    }

    /**
     * Toggle active status.
     */
    public function toggleActive(EmailTemplate $emailTemplate)
    {
        $emailTemplate->update(['is_active' => !$emailTemplate->is_active]);

        return back()->with('success', 'Email template status updated.');
    }

    /**
     * Preview the template.
     */
    public function preview(EmailTemplate $emailTemplate, Request $request)
    {
        $variables = $request->get('variables', []);
        $rendered = $emailTemplate->render($variables);

        return response()->json([
            'subject' => $rendered['subject'],
            'content' => $rendered['content'],
        ]);
    }

    /**
     * Send test email.
     */
    public function sendTest(EmailTemplate $emailTemplate, Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'variables' => 'nullable|array',
        ]);

        // Here you would implement the actual email sending logic
        // For now, just return success
        return back()->with('success', 'Test email sent successfully.');
    }
}
