<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\FeaturedContent;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class FeaturedContentController extends Controller
{
    public function __construct()
    {
        $this->middleware('admin');
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $featuredContents = FeaturedContent::ordered()->paginate(20);

        return view('admin.featured-content.index', compact('featuredContents'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.featured-content.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'link_url' => 'nullable|url',
            'link_text' => 'nullable|string|max:255',
            'position' => 'required|in:hero,sidebar,footer,homepage',
            'sort_order' => 'integer|min:0',
            'is_active' => 'boolean',
            'expires_at' => 'nullable|date|after:now',
        ]);

        $data = $request->all();

        // Handle image upload
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('featured-content', 'public');
            $data['image_url'] = Storage::disk('public')->url($imagePath);
        }

        FeaturedContent::create($data);

        return redirect()->route('admin.featured-content.index')
            ->with('success', 'Featured content created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(FeaturedContent $featuredContent)
    {
        return view('admin.featured-content.show', compact('featuredContent'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(FeaturedContent $featuredContent)
    {
        return view('admin.featured-content.edit', compact('featuredContent'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, FeaturedContent $featuredContent)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'link_url' => 'nullable|url',
            'link_text' => 'nullable|string|max:255',
            'position' => 'required|in:hero,sidebar,footer,homepage',
            'sort_order' => 'integer|min:0',
            'is_active' => 'boolean',
            'expires_at' => 'nullable|date',
        ]);

        $data = $request->all();

        // Handle image upload
        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($featuredContent->image_url) {
                $oldPath = str_replace('/storage/', '', $featuredContent->image_url);
                Storage::disk('public')->delete($oldPath);
            }

            $imagePath = $request->file('image')->store('featured-content', 'public');
            $data['image_url'] = Storage::disk('public')->url($imagePath);
        }

        $featuredContent->update($data);

        return redirect()->route('admin.featured-content.index')
            ->with('success', 'Featured content updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(FeaturedContent $featuredContent)
    {
        // Delete associated image
        if ($featuredContent->image_url) {
            $imagePath = str_replace('/storage/', '', $featuredContent->image_url);
            Storage::disk('public')->delete($imagePath);
        }

        $featuredContent->delete();

        return redirect()->route('admin.featured-content.index')
            ->with('success', 'Featured content deleted successfully.');
    }

    /**
     * Toggle active status.
     */
    public function toggleActive(FeaturedContent $featuredContent)
    {
        $featuredContent->update(['is_active' => !$featuredContent->is_active]);

        return back()->with('success', 'Featured content status updated.');
    }

    /**
     * Update sort order.
     */
    public function updateOrder(Request $request)
    {
        $request->validate([
            'items' => 'required|array',
            'items.*.id' => 'required|integer|exists:featured_content,id',
            'items.*.sort_order' => 'required|integer|min:0',
        ]);

        foreach ($request->items as $item) {
            FeaturedContent::where('id', $item['id'])
                ->update(['sort_order' => $item['sort_order']]);
        }

        return response()->json(['success' => true]);
    }
}
