<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Job;
use App\Services\JobVerificationService;
use Illuminate\Http\Request;

class JobController extends Controller
{
    protected $verificationService;

    public function __construct(JobVerificationService $verificationService)
    {
        $this->middleware('admin');
        $this->verificationService = $verificationService;
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Job::with(['company', 'applications']);

        // Search functionality
        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhereHas('company', function ($companyQuery) use ($search) {
                      $companyQuery->where('name', 'like', "%{$search}%");
                  });
            });
        }

        // Filter by status
        if ($request->has('status') && !empty($request->status)) {
            $query->where('status', $request->status);
        }

        // Filter by verification status
        if ($request->has('verification_status') && !empty($request->verification_status)) {
            $query->where('verification_status', $request->verification_status);
        }

        // Filter by scam suspicion
        if ($request->has('scam_filter')) {
            if ($request->scam_filter === 'suspected') {
                $query->where('is_scam_suspected', true);
            } elseif ($request->scam_filter === 'clean') {
                $query->where('is_scam_suspected', false);
            }
        }

        $jobs = $query->orderBy('created_at', 'desc')->paginate(20);

        return view('admin.jobs.index', compact('jobs'));
    }

    /**
     * Display the specified resource.
     */
    public function show(Job $job)
    {
        $job->load(['company', 'applications.user.profile']);
        $analysis = $this->verificationService->analyzeJob($job);

        return view('admin.jobs.show', compact('job', 'analysis'));
    }

    /**
     * Update job status.
     */
    public function updateStatus(Request $request, Job $job)
    {
        $request->validate([
            'status' => 'required|in:active,pending,rejected,expired',
        ]);

        $job->update(['status' => $request->status]);

        return back()->with('success', 'Job status updated successfully.');
    }

    /**
     * Update verification status.
     */
    public function updateVerification(Request $request, Job $job)
    {
        $request->validate([
            'verification_status' => 'required|in:pending,verified,rejected',
            'moderation_notes' => 'nullable|string|max:1000',
        ]);

        $data = ['verification_status' => $request->verification_status];

        if ($request->verification_status === 'verified') {
            $data['verified_at'] = now();
            $data['is_scam_suspected'] = false;
        } elseif ($request->verification_status === 'rejected') {
            $data['moderated_at'] = now();
            $data['moderation_notes'] = $request->moderation_notes;
        }

        $job->update($data);

        return back()->with('success', 'Job verification status updated successfully.');
    }

    /**
     * Flag job as scam.
     */
    public function flagAsScam(Request $request, Job $job)
    {
        $request->validate([
            'flags' => 'required|array',
            'flags.*' => 'string',
        ]);

        $job->flagAsScam($request->flags);

        return back()->with('success', 'Job flagged as potential scam.');
    }

    /**
     * Remove scam flag.
     */
    public function removeScamFlag(Job $job)
    {
        $job->update([
            'is_scam_suspected' => false,
            'scam_flags' => null,
        ]);

        return back()->with('success', 'Scam flag removed from job.');
    }

    /**
     * Toggle featured status.
     */
    public function toggleFeatured(Job $job)
    {
        $job->update(['featured' => !$job->featured]);

        return back()->with('success', 'Job featured status updated.');
    }

    /**
     * Bulk actions for jobs.
     */
    public function bulkAction(Request $request)
    {
        $request->validate([
            'action' => 'required|in:activate,deactivate,verify,reject,delete,feature,unfeature',
            'job_ids' => 'required|array',
            'job_ids.*' => 'integer|exists:jobs,id',
        ]);

        $jobs = Job::whereIn('id', $request->job_ids)->get();

        switch ($request->action) {
            case 'activate':
                $jobs->each->update(['status' => 'active']);
                $message = 'Jobs activated successfully.';
                break;
            case 'deactivate':
                $jobs->each->update(['status' => 'pending']);
                $message = 'Jobs deactivated successfully.';
                break;
            case 'verify':
                $jobs->each(function ($job) {
                    $job->markAsVerified();
                });
                $message = 'Jobs verified successfully.';
                break;
            case 'reject':
                $jobs->each(function ($job) {
                    $job->markAsRejected('Bulk rejected by admin');
                });
                $message = 'Jobs rejected successfully.';
                break;
            case 'feature':
                $jobs->each->update(['featured' => true]);
                $message = 'Jobs featured successfully.';
                break;
            case 'unfeature':
                $jobs->each->update(['featured' => false]);
                $message = 'Jobs unfeatured successfully.';
                break;
            case 'delete':
                $jobs->each->delete();
                $message = 'Jobs deleted successfully.';
                break;
        }

        return back()->with('success', $message);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Job $job)
    {
        $job->delete();

        return redirect()->route('admin.jobs.index')
            ->with('success', 'Job deleted successfully.');
    }
}
