<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SystemUpdate;
use Illuminate\Http\Request;

class SystemUpdateController extends Controller
{
    public function __construct()
    {
        $this->middleware('admin');
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $systemUpdates = SystemUpdate::orderBy('created_at', 'desc')->paginate(20);

        return view('admin.system-updates.index', compact('systemUpdates'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.system-updates.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'version' => 'nullable|string|max:50',
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'type' => 'required|in:feature,bugfix,security,maintenance,announcement',
            'severity' => 'required|in:low,medium,high,critical',
            'is_published' => 'boolean',
            'published_at' => 'nullable|date',
        ]);

        SystemUpdate::create($request->all());

        return redirect()->route('admin.system-updates.index')
            ->with('success', 'System update created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(SystemUpdate $systemUpdate)
    {
        return view('admin.system-updates.show', compact('systemUpdate'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(SystemUpdate $systemUpdate)
    {
        return view('admin.system-updates.edit', compact('systemUpdate'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, SystemUpdate $systemUpdate)
    {
        $request->validate([
            'version' => 'nullable|string|max:50',
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'type' => 'required|in:feature,bugfix,security,maintenance,announcement',
            'severity' => 'required|in:low,medium,high,critical',
            'is_published' => 'boolean',
            'published_at' => 'nullable|date',
        ]);

        $systemUpdate->update($request->all());

        return redirect()->route('admin.system-updates.index')
            ->with('success', 'System update updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(SystemUpdate $systemUpdate)
    {
        $systemUpdate->delete();

        return redirect()->route('admin.system-updates.index')
            ->with('success', 'System update deleted successfully.');
    }

    /**
     * Toggle published status.
     */
    public function togglePublished(SystemUpdate $systemUpdate)
    {
        if (!$systemUpdate->is_published) {
            $systemUpdate->publish();
        } else {
            $systemUpdate->update(['is_published' => false, 'published_at' => null]);
        }

        return back()->with('success', 'System update published status updated.');
    }

    /**
     * Get published updates for API.
     */
    public function published()
    {
        $updates = SystemUpdate::published()
            ->orderBy('published_at', 'desc')
            ->get();

        return response()->json($updates);
    }
}
