<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Job;
use App\Models\Application;
use App\Models\Organization;
use App\Models\SystemSetting;
use App\Models\ActivityLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Response;

class AdminController extends Controller
{
    public function __construct()
    {
        $this->middleware('admin')->except(['login', 'authenticate', 'logout']);
    }

    protected function isAdmin($user)
    {
        return $user && $user->role === 'admin';
    }

    public function login()
    {
        if (auth()->check() && $this->isAdmin(auth()->user())) {
            return redirect()->route('admin.dashboard');
        }

        return view('admin.login');
    }

    public function authenticate(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        $credentials = $request->only('email', 'password');

        // Check if admin user exists
        $adminUser = User::where('email', $credentials['email'])
            ->where('role', 'admin')
            ->first();

        if (!$adminUser) {
            return back()->withErrors([
                'email' => 'These credentials do not match our admin records.',
            ])->onlyInput('email');
        }

        // Attempt to authenticate
        if (auth()->attempt($credentials, $request->boolean('remember'))) {
            $request->session()->regenerate();
            
            // Log admin login
            ActivityLog::create([
                'user_id' => auth()->id(),
                'action' => 'login',
                'resource_type' => 'admin',
                'description' => 'Admin user logged in successfully',
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ]);

            return redirect()->intended(route('admin.dashboard'));
        }

        return back()->withErrors([
            'email' => 'These credentials do not match our records.',
        ])->onlyInput('email');
    }

    public function logout(Request $request)
    {
        if (auth()->check()) {
            // Log admin logout
            ActivityLog::create([
                'user_id' => auth()->id(),
                'action' => 'logout',
                'resource_type' => 'admin',
                'description' => 'Admin user logged out',
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ]);
        }

        auth()->logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('admin.login')->with('status', 'You have been logged out successfully.');
    }

    public function dashboard()
    {
        // Platform statistics
        $stats = [
            'total_users' => User::count(),
            'active_jobs' => Job::where('status', 'active')->count(),
            'total_applications' => Application::count(),
            'revenue' => 0, // Placeholder for revenue calculation
        ];

        // Recent activities (placeholder data)
        $recentActivities = [
            [
                'user' => 'John Doe',
                'action' => 'Applied for Senior Developer position',
                'description' => 'at TechCorp Cameroon',
                'time' => '2 minutes ago'
            ],
            [
                'user' => 'Sarah Nkou',
                'action' => 'Posted new job',
                'description' => 'Marketing Manager position',
                'time' => '15 minutes ago'
            ],
            [
                'user' => 'Admin',
                'action' => 'Approved job posting',
                'description' => 'Data Analyst position reviewed',
                'time' => '1 hour ago'
            ],
            [
                'user' => 'Marie-Claire Ngo',
                'action' => 'Updated company profile',
                'description' => 'Added new team information',
                'time' => '2 hours ago'
            ],
        ];

        return view('admin.dashboard', compact('stats', 'recentActivities'));
    }

    public function users()
    {
        $users = User::with('organization')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.users.index', compact('users'));
    }

    public function jobs()
    {
        $jobs = Job::with(['organization', 'applications'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.jobs.index', compact('jobs'));
    }

    public function applications()
    {
        $applications = Application::with(['user', 'job.organization'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.applications.index', compact('applications'));
    }

    public function analytics()
    {
        // Analytics data - in production, this would come from a proper analytics service
        $analyticsData = [
            'userGrowth' => 24,
            'totalJobs' => 156,
            'totalApplications' => 892,
            'conversionRate' => 68,
            'topCompanies' => [
                ['name' => 'TechCorp Cameroon', 'applications' => 156, 'matchRate' => 89],
                ['name' => 'FinancePlus', 'applications' => 98, 'matchRate' => 76],
                ['name' => 'MediCare Ltd', 'applications' => 87, 'matchRate' => 82],
            ],
            'systemHealth' => [
                'uptime' => '99.9%',
                'dbPerformance' => 'Excellent',
                'responseTime' => '245ms',
                'activeSessions' => 1247,
                'errorRate' => '0.01%',
            ],
            'jobCategories' => [
                ['name' => 'Technology', 'percentage' => 85],
                ['name' => 'Finance', 'percentage' => 72],
                ['name' => 'Healthcare', 'percentage' => 58],
                ['name' => 'Retail', 'percentage' => 45],
            ]
        ];

        return view('admin.analytics', compact('analyticsData'));
    }

    public function updateUserStatus(Request $request, User $user)
    {
        $request->validate([
            'status' => 'required|in:active,suspended,banned',
        ]);

        $user->update(['status' => $request->status]);

        return back()->with('success', 'User status updated successfully.');
    }

    public function updateJobStatus(Request $request, Job $job)
    {
        $request->validate([
            'status' => 'required|in:active,pending,rejected,expired',
        ]);

        $job->update(['status' => $request->status]);

        return back()->with('success', 'Job status updated successfully.');
    }

    public function deleteJob(Job $job)
    {
        $job->delete();

        return back()->with('success', 'Job deleted successfully.');
    }

    public function deleteUser(User $user)
    {
        // Prevent deleting admin users
        if ($this->isAdmin($user)) {
            return back()->with('error', 'Cannot delete admin user.');
        }

        $user->delete();

        return back()->with('success', 'User deleted successfully.');
    }

    public function settings()
    {
        $settings = [
            // General Settings
            'site_name' => SystemSetting::get('site_name', config('app.name')),
            'site_description' => SystemSetting::get('site_description', 'Job platform for Cameroon'),
            'contact_email' => SystemSetting::get('contact_email', 'admin@hirecameroon.com'),
            'default_currency' => SystemSetting::get('default_currency', 'XAF'),

            // Platform Features
            'registration_enabled' => SystemSetting::get('registration_enabled', true),
            'job_posting_enabled' => SystemSetting::get('job_posting_enabled', true),
            'email_notifications' => SystemSetting::get('email_notifications', true),
            'featured_jobs_enabled' => SystemSetting::get('featured_jobs_enabled', true),

            // Maintenance & Security
            'maintenance_mode' => SystemSetting::get('maintenance_mode', false),
            'debug_mode' => SystemSetting::get('debug_mode', false),
            'session_timeout' => SystemSetting::get('session_timeout', 1440),
            'max_login_attempts' => SystemSetting::get('max_login_attempts', 5),

            // Email Configuration
            'smtp_host' => SystemSetting::get('smtp_host', ''),
            'smtp_port' => SystemSetting::get('smtp_port', 587),
            'smtp_username' => SystemSetting::get('smtp_username', ''),
            'smtp_encryption' => SystemSetting::get('smtp_encryption', 'tls'),

            // Appearance & Branding
            'logo_url' => SystemSetting::get('logo_url', ''),
            'favicon_url' => SystemSetting::get('favicon_url', ''),
            'primary_color' => SystemSetting::get('primary_color', '#3B82F6'),
            'secondary_color' => SystemSetting::get('secondary_color', '#6B7280'),

            // Subscription Plans
            'subscription_plans' => SystemSetting::get('subscription_plans', [
                'basic' => ['name' => 'Basic', 'price' => 0, 'features' => ['5 jobs/month', 'Basic support']],
                'premium' => ['name' => 'Premium', 'price' => 50000, 'features' => ['Unlimited jobs', 'Priority support', 'Analytics']],
                'enterprise' => ['name' => 'Enterprise', 'price' => 150000, 'features' => ['All features', 'Dedicated support', 'Custom branding']]
            ]),

            // Backup Settings
            'auto_backup_enabled' => SystemSetting::get('auto_backup_enabled', false),
            'backup_frequency' => SystemSetting::get('backup_frequency', 'daily'),
            'backup_retention_days' => SystemSetting::get('backup_retention_days', 30),

            // Platform Customization
            'welcome_message' => SystemSetting::get('welcome_message', 'Welcome to Hire Cameroon'),
            'footer_text' => SystemSetting::get('footer_text', '© 2025 Hire Cameroon. All rights reserved.'),
            'social_links' => SystemSetting::get('social_links', []),
            'custom_css' => SystemSetting::get('custom_css', ''),
        ];

        return view('admin.settings', compact('settings'));
    }

    public function createBackup()
    {
        try {
            $timestamp = now()->format('Y-m-d_H-i-s');
            $backupPath = "backups/{$timestamp}";

            // Create backup directory if it doesn't exist
            Storage::disk('local')->makeDirectory($backupPath);

            // Backup database
            $dbBackupPath = "{$backupPath}/database.sql";
            $this->backupDatabase($dbBackupPath);

            // Backup uploaded files
            $filesBackupPath = "{$backupPath}/files";
            $this->backupFiles($filesBackupPath);

            // Create backup metadata
            $metadata = [
                'timestamp' => now(),
                'version' => config('app.version', '1.0.0'),
                'database_size' => $this->getDatabaseSize(),
                'files_count' => $this->getFilesCount(),
            ];

            Storage::disk('local')->put("{$backupPath}/metadata.json", json_encode($metadata, JSON_PRETTY_PRINT));

            // Log backup creation
            Log::info("System backup created: {$backupPath}");

            return back()->with('success', 'Backup created successfully.');

        } catch (\Exception $e) {
            Log::error('Backup creation failed: ' . $e->getMessage());
            return back()->with('error', 'Backup creation failed: ' . $e->getMessage());
        }
    }

    public function listBackups()
    {
        $backups = [];
        $backupDir = 'backups';

        if (Storage::disk('local')->exists($backupDir)) {
            $directories = Storage::disk('local')->directories($backupDir);

            foreach ($directories as $dir) {
                $metadataPath = "{$dir}/metadata.json";
                if (Storage::disk('local')->exists($metadataPath)) {
                    $metadata = json_decode(Storage::disk('local')->get($metadataPath), true);
                    $backups[] = [
                        'path' => $dir,
                        'name' => basename($dir),
                        'created_at' => $metadata['timestamp'] ?? null,
                        'version' => $metadata['version'] ?? 'Unknown',
                        'database_size' => $metadata['database_size'] ?? 0,
                        'files_count' => $metadata['files_count'] ?? 0,
                        'size' => $this->getDirectorySize($dir),
                    ];
                }
            }
        }

        // Sort by creation date (newest first)
        usort($backups, function($a, $b) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        });

        return view('admin.backups.index', compact('backups'));
    }

    public function downloadBackup($backupName)
    {
        $backupPath = "backups/{$backupName}";

        if (!Storage::disk('local')->exists($backupPath)) {
            return back()->with('error', 'Backup not found.');
        }

        // Create a zip file
        $zipFileName = "backup-{$backupName}.zip";
        $zipPath = "temp/{$zipFileName}";

        $this->createZipFromDirectory($backupPath, $zipPath);

        return Response::download(storage_path('app/' . $zipPath), $zipFileName);
    }

    public function deleteBackup($backupName)
    {
        $backupPath = "backups/{$backupName}";

        if (!Storage::disk('local')->exists($backupPath)) {
            return back()->with('error', 'Backup not found.');
        }

        Storage::disk('local')->deleteDirectory($backupPath);

        Log::info("Backup deleted: {$backupName}");

        return back()->with('success', 'Backup deleted successfully.');
    }

    private function backupDatabase($path)
    {
        // This is a simplified database backup
        // In production, you'd use mysqldump or similar
        $tables = [
            'users', 'companies', 'jobs', 'applications', 'resumes',
            'subscriptions', 'payment_transactions', 'system_settings'
        ];

        $sql = "-- Hire Cameroon Database Backup\n";
        $sql .= "-- Created: " . now() . "\n\n";

        foreach ($tables as $table) {
            try {
                $createTable = DB::select("SHOW CREATE TABLE {$table}");
                if (!empty($createTable)) {
                    $sql .= $createTable[0]->{'Create Table'} . ";\n\n";

                    $records = DB::table($table)->get();
                    if ($records->count() > 0) {
                        foreach ($records as $record) {
                            $columns = array_keys((array) $record);
                            $values = array_map(function($value) {
                                return is_null($value) ? 'NULL' : "'" . addslashes($value) . "'";
                            }, (array) $record);

                            $sql .= "INSERT INTO {$table} (" . implode(', ', $columns) . ") VALUES (" . implode(', ', $values) . ");\n";
                        }
                        $sql .= "\n";
                    }
                }
            } catch (\Exception $e) {
                // Table might not exist, skip
                continue;
            }
        }

        Storage::disk('local')->put($path, $sql);
    }

    private function backupFiles($path)
    {
        // Copy important uploaded files
        $directoriesToBackup = [
            'public/logos',
            'public/favicons',
            'public/resumes',
            'public/company-logos',
        ];

        foreach ($directoriesToBackup as $dir) {
            if (Storage::disk('public')->exists($dir)) {
                $files = Storage::disk('public')->allFiles($dir);
                foreach ($files as $file) {
                    $destination = str_replace('public/', '', $file);
                    Storage::disk('local')->put("{$path}/{$destination}", Storage::disk('public')->get($file));
                }
            }
        }
    }

    private function getDatabaseSize()
    {
        try {
            $result = DB::select("SELECT ROUND(SUM(data_length + index_length) / 1024 / 1024, 2) as size_mb FROM information_schema.tables WHERE table_schema = ?", [config('database.connections.mysql.database')]);
            return $result[0]->size_mb ?? 0;
        } catch (\Exception $e) {
            return 0;
        }
    }

    private function getFilesCount()
    {
        $count = 0;
        $directories = ['public/logos', 'public/favicons', 'public/resumes', 'public/company-logos'];

        foreach ($directories as $dir) {
            if (Storage::disk('public')->exists($dir)) {
                $count += count(Storage::disk('public')->allFiles($dir));
            }
        }

        return $count;
    }

    private function getDirectorySize($path)
    {
        $size = 0;
        $files = Storage::disk('local')->allFiles($path);

        foreach ($files as $file) {
            $size += Storage::disk('local')->size($file);
        }

        return $this->formatBytes($size);
    }

    private function formatBytes($bytes)
    {
        $units = ['B', 'KB', 'MB', 'GB'];
        $i = 0;

        while ($bytes >= 1024 && $i < count($units) - 1) {
            $bytes /= 1024;
            $i++;
        }

        return round($bytes, 2) . ' ' . $units[$i];
    }

    private function createZipFromDirectory($source, $destination)
    {
        // This is a simplified zip creation
        // In production, you'd use ZipArchive or similar
        $zip = new \ZipArchive();
        $zip->open(storage_path('app/' . $destination), \ZipArchive::CREATE | \ZipArchive::OVERWRITE);

        $files = Storage::disk('local')->allFiles($source);

        foreach ($files as $file) {
            $zip->addFile(storage_path('app/' . $file), $file);
        }

        $zip->close();
    }

    public function userActivity()
    {
        return view('admin.user-activity');
    }

    public function getUserActivityData(Request $request)
    {
        $days = $request->get('days', 30);
        $stats = \App\Services\UserActivityService::getActivityStats($days);

        return response()->json([
            'stats' => $stats,
            'daily_activity' => $stats['daily_activity'],
            'top_actions' => $stats['top_actions']
        ]);
    }

    public function getUserActivityFeed(Request $request)
    {
        $days = $request->get('days', 30);
        $page = $request->get('page', 1);
        $perPage = $request->get('per_page', 20);

        $activities = ActivityLog::with('user')
            ->recent($days)
            ->orderBy('created_at', 'desc')
            ->paginate($perPage, ['*'], 'page', $page);

        // Add resource names for better display
        $activities->getCollection()->transform(function ($activity) {
            if ($activity->resource_type && $activity->resource_id) {
                $activity->resource_name = $this->getResourceName($activity->resource_type, $activity->resource_id);
            }
            return $activity;
        });

        return response()->json([
            'activities' => $activities->items(),
            'has_more' => $activities->hasMorePages()
        ]);
    }

    public function systemHealth()
    {
        return view('admin.system-health');
    }

    public function getSystemHealthData(Request $request)
    {
        $hours = $request->get('hours', 24);
        $healthData = \App\Services\SystemHealthService::getLatestHealthData();
        $healthHistory = \App\Services\SystemHealthService::getHealthHistory($hours);
        $healthStatus = \App\Services\SystemHealthService::getHealthStatus();

        return response()->json([
            'current' => $healthData,
            'history' => $healthHistory,
            'status' => $healthStatus
        ]);
    }

    public function auditLogs()
    {
        return view('admin.audit-logs');
    }

    public function getAuditLogsData(Request $request)
    {
        $filters = [
            'days' => $request->get('days', 30),
            'action' => $request->get('action'),
            'resource_type' => $request->get('resource'),
            'resource_id' => $request->get('resource_id'),
            'user_id' => $request->get('user_id'),
            'search' => $request->get('search'),
        ];

        $logs = \App\Services\AuditService::getAuditLogs($filters, $request->get('per_page', 20));

        return response()->json([
            'logs' => $logs->items(),
            'pagination' => [
                'current_page' => $logs->currentPage(),
                'last_page' => $logs->lastPage(),
                'per_page' => $logs->perPage(),
                'total' => $logs->total(),
                'from' => $logs->firstItem(),
                'to' => $logs->lastItem(),
                'prev_page_url' => $logs->previousPageUrl(),
                'next_page_url' => $logs->nextPageUrl(),
            ]
        ]);
    }

    public function getAuditLogDetails($id)
    {
        $log = ActivityLog::with('user')->findOrFail($id);

        return response()->json(['log' => $log]);
    }

    public function exportAuditLogs(Request $request)
    {
        $filters = [
            'days' => $request->get('days', 30),
            'action' => $request->get('action'),
            'resource_type' => $request->get('resource'),
            'resource_id' => $request->get('resource_id'),
            'user_id' => $request->get('user_id'),
            'search' => $request->get('search'),
        ];

        $csvData = \App\Services\AuditService::exportAuditLogs($filters);

        $filename = 'audit-logs-' . now()->format('Y-m-d-H-i-s') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function() use ($csvData) {
            $file = fopen('php://output', 'w');
            foreach ($csvData as $row) {
                fputcsv($file, $row);
            }
            fclose($file);
        };

        return Response::stream($callback, 200, $headers);
    }

    private function getResourceName($type, $id)
    {
        try {
            switch ($type) {
                case 'job':
                    $resource = Job::find($id);
                    return $resource ? $resource->title : 'Unknown Job';
                case 'company':
                    $resource = Organization::find($id);
                    return $resource ? $resource->name : 'Unknown Company';
                case 'application':
                    $resource = Application::find($id);
                    return $resource ? "Application #{$id}" : 'Unknown Application';
                default:
                    return null;
            }
        } catch (\Exception $e) {
            return null;
        }
    }


    public function updateSettings(Request $request)
    {
        $request->validate([
            // General Settings
            'site_name' => 'required|string|max:255',
            'site_description' => 'nullable|string|max:500',
            'contact_email' => 'required|email',
            'default_currency' => 'required|in:XAF,USD,EUR',

            // Platform Features
            'registration_enabled' => 'boolean',
            'job_posting_enabled' => 'boolean',
            'email_notifications' => 'boolean',
            'featured_jobs_enabled' => 'boolean',

            // Maintenance & Security
            'maintenance_mode' => 'boolean',
            'debug_mode' => 'boolean',
            'session_timeout' => 'required|integer|min:30|max:10080',
            'max_login_attempts' => 'required|integer|min:3|max:20',

            // Email Configuration
            'smtp_host' => 'nullable|string|max:255',
            'smtp_port' => 'nullable|integer|min:1|max:65535',
            'smtp_username' => 'nullable|string|max:255',
            'smtp_encryption' => 'required|in:tls,ssl,none',

            // Appearance & Branding
            'primary_color' => 'nullable|string|regex:/^#[a-fA-F0-9]{6}$/',
            'secondary_color' => 'nullable|string|regex:/^#[a-fA-F0-9]{6}$/',

            // Backup Settings
            'auto_backup_enabled' => 'boolean',
            'backup_frequency' => 'required|in:daily,weekly,monthly',
            'backup_retention_days' => 'required|integer|min:1|max:365',

            // Platform Customization
            'welcome_message' => 'nullable|string|max:500',
            'footer_text' => 'nullable|string|max:500',
            'custom_css' => 'nullable|string',
        ]);

        // Handle file uploads for logo and favicon
        if ($request->hasFile('logo')) {
            $logoPath = $request->file('logo')->store('logos', 'public');
            SystemSetting::set('logo_url', $logoPath, 'file', 'appearance', 'Site logo image');
        }

        if ($request->hasFile('favicon')) {
            $faviconPath = $request->file('favicon')->store('favicons', 'public');
            SystemSetting::set('favicon_url', $faviconPath, 'file', 'appearance', 'Site favicon');
        }

        // Handle social links (JSON)
        if ($request->has('social_links')) {
            $socialLinks = [];
            foreach ($request->social_links as $platform => $url) {
                if (!empty($url)) {
                    $socialLinks[$platform] = $url;
                }
            }
            SystemSetting::set('social_links', $socialLinks, 'json', 'customization', 'Social media links');
        }

        // Handle subscription plans (JSON)
        if ($request->has('subscription_plans')) {
            SystemSetting::set('subscription_plans', $request->subscription_plans, 'json', 'subscription', 'Available subscription plans');
        }

        // Update all other settings
        $settingsToUpdate = [
            // General
            'site_name' => ['value' => $request->site_name, 'group' => 'general', 'description' => 'Site name displayed in header'],
            'site_description' => ['value' => $request->site_description, 'group' => 'general', 'description' => 'Site description for SEO'],
            'contact_email' => ['value' => $request->contact_email, 'group' => 'general', 'description' => 'Primary contact email'],
            'default_currency' => ['value' => $request->default_currency, 'group' => 'general', 'description' => 'Default currency for pricing'],

            // Platform Features
            'registration_enabled' => ['value' => $request->boolean('registration_enabled'), 'type' => 'boolean', 'group' => 'features', 'description' => 'Allow new user registrations'],
            'job_posting_enabled' => ['value' => $request->boolean('job_posting_enabled'), 'type' => 'boolean', 'group' => 'features', 'description' => 'Allow companies to post jobs'],
            'email_notifications' => ['value' => $request->boolean('email_notifications'), 'type' => 'boolean', 'group' => 'features', 'description' => 'Enable email notifications'],
            'featured_jobs_enabled' => ['value' => $request->boolean('featured_jobs_enabled'), 'type' => 'boolean', 'group' => 'features', 'description' => 'Enable featured jobs feature'],

            // Maintenance & Security
            'maintenance_mode' => ['value' => $request->boolean('maintenance_mode'), 'type' => 'boolean', 'group' => 'maintenance', 'description' => 'Put site in maintenance mode'],
            'debug_mode' => ['value' => $request->boolean('debug_mode'), 'type' => 'boolean', 'group' => 'maintenance', 'description' => 'Enable debug mode'],
            'session_timeout' => ['value' => $request->session_timeout, 'type' => 'integer', 'group' => 'security', 'description' => 'Session timeout in minutes'],
            'max_login_attempts' => ['value' => $request->max_login_attempts, 'type' => 'integer', 'group' => 'security', 'description' => 'Maximum login attempts before lockout'],

            // Email Configuration
            'smtp_host' => ['value' => $request->smtp_host, 'group' => 'email', 'description' => 'SMTP server hostname'],
            'smtp_port' => ['value' => $request->smtp_port, 'type' => 'integer', 'group' => 'email', 'description' => 'SMTP server port'],
            'smtp_username' => ['value' => $request->smtp_username, 'group' => 'email', 'description' => 'SMTP authentication username'],
            'smtp_encryption' => ['value' => $request->smtp_encryption, 'group' => 'email', 'description' => 'SMTP encryption type'],

            // Appearance & Branding
            'primary_color' => ['value' => $request->primary_color, 'group' => 'appearance', 'description' => 'Primary brand color'],
            'secondary_color' => ['value' => $request->secondary_color, 'group' => 'appearance', 'description' => 'Secondary brand color'],

            // Backup Settings
            'auto_backup_enabled' => ['value' => $request->boolean('auto_backup_enabled'), 'type' => 'boolean', 'group' => 'backup', 'description' => 'Enable automatic backups'],
            'backup_frequency' => ['value' => $request->backup_frequency, 'group' => 'backup', 'description' => 'How often to create backups'],
            'backup_retention_days' => ['value' => $request->backup_retention_days, 'type' => 'integer', 'group' => 'backup', 'description' => 'How long to keep backups'],

            // Platform Customization
            'welcome_message' => ['value' => $request->welcome_message, 'group' => 'customization', 'description' => 'Welcome message on homepage'],
            'footer_text' => ['value' => $request->footer_text, 'group' => 'customization', 'description' => 'Footer copyright text'],
            'custom_css' => ['value' => $request->custom_css, 'group' => 'customization', 'description' => 'Custom CSS for styling'],
        ];

        foreach ($settingsToUpdate as $key => $config) {
            SystemSetting::set(
                $key,
                $config['value'],
                $config['type'] ?? 'string',
                $config['group'] ?? 'general',
                $config['description'] ?? null
            );
        }

        // Clear cache to ensure fresh data
        SystemSetting::clearCache();

        return back()->with('success', 'Settings updated successfully.');
    }

    public function bulkEmail()
    {
        // Campaign statistics
        $stats = [
            'total_campaigns' => \App\Models\EmailCampaign::count(),
            'total_emails_sent' => \App\Models\EmailCampaign::sum('emails_sent'),
            'average_open_rate' => \App\Models\EmailCampaign::where('emails_sent', '>', 0)->avg('emails_opened') * 100 / \App\Models\EmailCampaign::where('emails_sent', '>', 0)->avg('emails_sent') ?: 0,
            'average_click_rate' => \App\Models\EmailCampaign::where('emails_sent', '>', 0)->avg('emails_clicked') * 100 / \App\Models\EmailCampaign::where('emails_sent', '>', 0)->avg('emails_sent') ?: 0,
        ];

        return view('admin.bulk-email', compact('stats'));
    }

    public function getBulkEmailData(Request $request)
    {
        $status = $request->get('status', '');
        $page = $request->get('page', 1);
        $perPage = 20;

        $query = \App\Models\EmailCampaign::with(['creator', 'template'])
            ->orderBy('created_at', 'desc');

        if ($status) {
            $query->where('status', $status);
        }

        $campaigns = $query->paginate($perPage, ['*'], 'page', $page);

        return response()->json([
            'campaigns' => $campaigns->items(),
            'pagination' => [
                'current_page' => $campaigns->currentPage(),
                'last_page' => $campaigns->lastPage(),
                'per_page' => $campaigns->perPage(),
                'total' => $campaigns->total(),
                'from' => $campaigns->firstItem(),
                'to' => $campaigns->lastItem(),
                'prev_page_url' => $campaigns->previousPageUrl(),
                'next_page_url' => $campaigns->nextPageUrl(),
            ]
        ]);
    }

    public function createEmailCampaign(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'subject' => 'required|string|max:255',
            'content' => 'required|string',
            'template_id' => 'nullable|exists:email_templates,id',
            'recipients' => 'required|array',
            'scheduled_at' => 'nullable|date|after:now',
        ]);

        // Calculate recipient count based on groups
        $recipientCount = $this->calculateRecipientCount($request->recipients);

        $campaign = \App\Models\EmailCampaign::create([
            'name' => $request->name,
            'subject' => $request->subject,
            'content' => $request->input('content'),
            'template_id' => $request->template_id,
            'recipient_groups' => $request->recipients,
            'recipient_count' => $recipientCount,
            'status' => $request->scheduled_at ? 'scheduled' : 'draft',
            'scheduled_at' => $request->scheduled_at,
            'created_by' => auth()->id(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Campaign created successfully',
            'campaign' => $campaign
        ]);
    }

    public function deleteEmailCampaign($campaignId)
    {
        $campaign = \App\Models\EmailCampaign::findOrFail($campaignId);

        // Only allow deletion of draft or scheduled campaigns
        if (!in_array($campaign->status, ['draft', 'scheduled'])) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete a campaign that has already been sent'
            ], 400);
        }

        $campaign->delete();

        return response()->json([
            'success' => true,
            'message' => 'Campaign deleted successfully'
        ]);
    }

    private function calculateRecipientCount(array $groups)
    {
        $count = 0;

        if (in_array('all_users', $groups)) {
            $count = \App\Models\User::count();
        } else {
            $query = \App\Models\User::query();

            if (in_array('job_seekers', $groups)) {
                $query->whereDoesntHave('organization');
            }

            if (in_array('employers', $groups)) {
                $query->orWhereHas('organization');
            }

            if (in_array('inactive_users', $groups)) {
                $query->where('last_login_at', '<', now()->subDays(30));
            }

            $count = $query->count();
        }

        return $count;
    }

    public function contentModeration()
    {
        // Moderation statistics
        $stats = [
            'pending_reviews' => Job::where('status', 'pending')->count() +
                                Organization::where('status', 'pending')->count() +
                                Application::where('status', 'pending')->count(),
            'approved_today' => ActivityLog::where('action', 'approve')
                                ->whereDate('created_at', today())
                                ->count(),
            'rejected_today' => ActivityLog::where('action', 'reject')
                                ->whereDate('created_at', today())
                                ->count(),
            'avg_review_time' => '5m', // Placeholder - would calculate actual average
        ];

        return view('admin.content-moderation', compact('stats'));
    }

    public function getContentModerationData(Request $request)
    {
        $type = $request->get('type', 'jobs');
        $status = $request->get('status', 'pending');

        $query = null;

        switch ($type) {
            case 'jobs':
                $query = Job::with(['organization', 'user']);
                break;
            case 'companies':
                $query = Organization::with('user');
                break;
            case 'applications':
                $query = Application::with(['job.organization', 'user']);
                break;
            default:
                return response()->json(['content' => []]);
        }

        if ($status !== 'all') {
            $query->where('status', $status);
        }

        $content = $query->orderBy('created_at', 'desc')->limit(50)->get();

        return response()->json(['content' => $content]);
    }

    public function getContentDetails($type, $id)
    {
        $model = null;

        switch ($type) {
            case 'job':
                $model = Job::with(['organization', 'user']);
                break;
            case 'company':
                $model = Organization::with('user');
                break;
            case 'application':
                $model = Application::with(['job.organization', 'user']);
                break;
            default:
                abort(404);
        }

        $content = $model->findOrFail($id);

        return response()->json(['content' => $content]);
    }

    public function seoManagement()
    {
        // SEO statistics (placeholder data)
        $seoStats = [
            'organic_traffic' => 15420,
            'keyword_rankings' => 45,
            'backlinks' => 234,
            'page_speed' => 85,
        ];

        // Meta tags for different pages
        $metaTags = [
            'home' => [
                'title' => 'HireCameroon - Find Work and Hire Faster in Cameroon',
                'description' => 'Find work and hire faster in Cameroon with HireCameroon - bilingual jobs, ATS, and mobile payments.',
            ],
            'jobs' => [
                'title' => 'Job Opportunities in Cameroon - HireCameroon',
                'description' => 'Browse thousands of job opportunities in Cameroon. Find your next career move with HireCameroon.',
            ],
            'companies' => [
                'title' => 'Top Companies Hiring in Cameroon - HireCameroon',
                'description' => 'Discover top companies hiring in Cameroon. Connect with leading employers across Douala, Yaoundé, and beyond.',
            ],
        ];

        // Keywords data
        $keywords = [
            'primary' => [
                ['term' => 'jobs Cameroon', 'volume' => 5400, 'rank' => 3],
                ['term' => 'hiring Cameroon', 'volume' => 2900, 'rank' => 2],
                ['term' => 'work Cameroon', 'volume' => 3600, 'rank' => 5],
            ],
            'secondary' => [
                ['term' => 'Cameroon employment', 'volume' => 1800, 'rank' => 7],
                ['term' => 'jobs Douala', 'volume' => 1200, 'rank' => 4],
                ['term' => 'jobs Yaoundé', 'volume' => 980, 'rank' => 6],
            ],
        ];

        // Sitemap status
        $sitemapStatus = [
            'last_generated' => '2024-11-02 10:30:00',
            'total_urls' => 1250,
        ];

        // Analytics data
        $analytics = [
            'top_pages' => [
                ['url' => '/', 'views' => 15420, 'bounce_rate' => 35],
                ['url' => '/jobs', 'views' => 8920, 'bounce_rate' => 42],
                ['url' => '/companies', 'views' => 5430, 'bounce_rate' => 38],
            ],
            'search_queries' => [
                ['term' => 'jobs Cameroon', 'position' => 3, 'clicks' => 1250],
                ['term' => 'hiring Cameroon', 'position' => 2, 'clicks' => 890],
                ['term' => 'work Cameroon', 'position' => 5, 'clicks' => 654],
            ],
        ];

        return view('admin.seo-management', compact('seoStats', 'metaTags', 'keywords', 'sitemapStatus', 'analytics'));
    }

    public function updateMetaTags(Request $request)
    {
        $request->validate([
            'page' => 'required|in:home,jobs,companies',
            'title' => 'required|string|max:60',
            'description' => 'required|string|max:160',
            'keywords' => 'nullable|string',
        ]);

        $page = $request->page;
        $metaKey = "meta_{$page}";

        $metaData = [
            'title' => $request->title,
            'description' => $request->description,
            'keywords' => $request->keywords,
        ];

        SystemSetting::set($metaKey, $metaData, 'json', 'seo', "Meta tags for {$page} page");

        return response()->json([
            'success' => true,
            'message' => 'Meta tags updated successfully'
        ]);
    }

    public function addKeyword(Request $request)
    {
        $request->validate([
            'keyword' => 'required|string|max:255',
        ]);

        // In a real implementation, this would save to a keywords table
        // For now, we'll just return success
        return response()->json([
            'success' => true,
            'message' => 'Keyword added successfully'
        ]);
    }

    public function updateKeyword(Request $request, $keyword)
    {
        $request->validate([
            'keyword' => 'required|string|max:255',
        ]);

        // In a real implementation, this would update the keyword
        return response()->json([
            'success' => true,
            'message' => 'Keyword updated successfully'
        ]);
    }

    public function runKeywordResearch(Request $request)
    {
        // In a real implementation, this would call a keyword research API
        // For now, we'll simulate the process
        return response()->json([
            'success' => true,
            'message' => 'Keyword research completed successfully'
        ]);
    }

    public function generateSitemap(Request $request)
    {
        // In a real implementation, this would generate an XML sitemap
        return response()->json([
            'success' => true,
            'message' => 'Sitemap generated successfully'
        ]);
    }

    public function submitSitemap(Request $request)
    {
        // In a real implementation, this would submit to Google and Bing
        return response()->json([
            'success' => true,
            'message' => 'Sitemap submitted to search engines'
        ]);
    }

    public function getSEOAnalytics(Request $request)
    {
        $period = $request->get('period', 30);

        // In a real implementation, this would fetch from Google Analytics/Search Console
        $analytics = [
            'top_pages' => [
                ['url' => '/', 'views' => 15420, 'bounce_rate' => 35],
                ['url' => '/jobs', 'views' => 8920, 'bounce_rate' => 42],
            ],
            'search_queries' => [
                ['term' => 'jobs Cameroon', 'position' => 3, 'clicks' => 1250],
            ],
        ];

        return Response::json([
            'success' => true,
            'analytics' => $analytics
        ]);
    }

    public function moderateContent(Request $request, $type, $id, $action)
    {
        $model = null;
        $logResourceType = $type;

        switch ($type) {
            case 'job':
                $model = Job::findOrFail($id);
                break;
            case 'company':
                $model = Organization::findOrFail($id);
                break;
            case 'application':
                $model = Application::findOrFail($id);
                break;
            default:
                abort(404);
        }

        // Update status
        if ($action === 'approve') {
            $model->update(['status' => 'approved']);
        } elseif ($action === 'reject') {
            $model->update(['status' => 'rejected']);
        }

        // Log the moderation action
        ActivityLog::create([
            'user_id' => auth()->id(),
            'action' => $action,
            'resource_type' => $logResourceType,
            'resource_id' => $id,
            'description' => ucfirst($action) . "ed {$type} content",
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
        ]);

        return response()->json([
            'success' => true,
            'message' => ucfirst($type) . ' ' . $action . 'ed successfully'
        ]);
    }

    public function integrationManagement()
    {
        // Integration statistics (placeholder data)
        $integrationStats = [
            'active_apis' => 3,
            'api_calls_today' => 15420,
            'webhook_events' => 892,
            'failed_requests' => 5,
        ];

        return view('admin.integration-management', compact('integrationStats'));
    }

    public function getApiKeys(Request $request)
    {
        $apiKeys = \App\Models\ApiKey::with('user')
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function($key) {
                return [
                    'id' => $key->id,
                    'name' => $key->name,
                    'masked_key' => 'hc_' . str_repeat('*', 32) . substr($key->api_key, -4),
                    'permissions' => $key->permissions ?? ['read'],
                    'is_active' => $key->is_active ?? true,
                    'last_used_at' => $key->last_used_at,
                    'request_count' => $key->request_count ?? 0,
                ];
            });

        return response()->json(['apiKeys' => $apiKeys]);
    }

    public function createApiKey(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'permissions' => 'required|array',
            'rate_limit' => 'required|integer|min:1|max:10000',
        ]);

        $apiKey = \App\Models\ApiKey::create([
            'name' => $request->name,
            'api_key' => 'hc_' . bin2hex(random_bytes(32)),
            'permissions' => $request->permissions,
            'rate_limit' => $request->rate_limit,
            'is_active' => true,
            'user_id' => auth()->id(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'API key created successfully',
            'apiKey' => $apiKey->api_key
        ]);
    }

    public function toggleApiKey($keyId)
    {
        $apiKey = \App\Models\ApiKey::findOrFail($keyId);
        $apiKey->update(['is_active' => !$apiKey->is_active]);

        return response()->json([
            'success' => true,
            'message' => 'API key status updated successfully'
        ]);
    }

    public function deleteApiKey($keyId)
    {
        $apiKey = \App\Models\ApiKey::findOrFail($keyId);
        $apiKey->delete();

        return response()->json([
            'success' => true,
            'message' => 'API key deleted successfully'
        ]);
    }

    public function getWebhooks(Request $request)
    {
        // In a real implementation, this would fetch from a webhooks table
        $webhooks = [
            [
                'id' => 1,
                'name' => 'Application Webhook',
                'url' => 'https://api.example.com/webhooks/applications',
                'events' => ['application.created', 'application.updated'],
                'last_delivery' => now()->subMinutes(5)->toISOString(),
                'success_rate' => 98.5,
            ],
            [
                'id' => 2,
                'name' => 'Job Webhook',
                'url' => 'https://api.example.com/webhooks/jobs',
                'events' => ['job.created', 'job.updated'],
                'last_delivery' => now()->subHours(2)->toISOString(),
                'success_rate' => 95.2,
            ],
        ];

        return response()->json(['webhooks' => $webhooks]);
    }

    public function getIntegrationLogs(Request $request)
    {
        $filter = $request->get('filter', 'all');

        // In a real implementation, this would fetch from an integration_logs table
        $logs = [
            [
                'id' => 1,
                'type' => 'api',
                'status' => 'success',
                'message' => 'Job search API called successfully',
                'endpoint' => '/api/jobs/search',
                'response_time' => 245,
                'created_at' => now()->subMinutes(5)->toISOString(),
            ],
            [
                'id' => 2,
                'type' => 'webhook',
                'status' => 'success',
                'message' => 'Application created webhook delivered',
                'endpoint' => 'https://api.example.com/webhooks/applications',
                'response_time' => 120,
                'created_at' => now()->subMinutes(15)->toISOString(),
            ],
            [
                'id' => 3,
                'type' => 'api',
                'status' => 'error',
                'message' => 'Invalid API key provided',
                'endpoint' => '/api/applications',
                'error_details' => 'Authentication failed',
                'created_at' => now()->subHours(1)->toISOString(),
            ],
        ];

        // Filter logs based on type
        if ($filter !== 'all') {
            $logs = array_filter($logs, function($log) use ($filter) {
                if ($filter === 'error') {
                    return $log['status'] === 'error';
                }
                return $log['type'] === $filter;
            });
        }

        return response()->json(['logs' => array_values($logs)]);
    }

    public function testPaymentIntegration(Request $request)
    {
        // In a real implementation, this would test the payment gateway
        return response()->json([
            'success' => true,
            'message' => 'Payment integration test completed successfully'
        ]);
    }

    public function exportIntegrationLogs(Request $request)
    {
        $filter = $request->get('filter', 'all');

        // In a real implementation, this would generate a CSV export
        $filename = 'integration-logs-' . now()->format('Y-m-d-H-i-s') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function() use ($filter) {
            $file = fopen('php://output', 'w');

            // CSV headers
            fputcsv($file, [
                'Timestamp',
                'Type',
                'Status',
                'Message',
                'Endpoint',
                'Response Time',
                'Error Details'
            ]);

            // Sample data - in real implementation, fetch from database
            $logs = [
                [now()->subMinutes(5)->toISOString(), 'api', 'success', 'Job search API called', '/api/jobs/search', '245ms', ''],
                [now()->subMinutes(15)->toISOString(), 'webhook', 'success', 'Application webhook', '/webhooks/apps', '120ms', ''],
            ];

            foreach ($logs as $log) {
                fputcsv($file, $log);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}