<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Application;
use App\Models\Job;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Storage;

class ApplicationController extends Controller
{
    /**
     * Apply for a job
     */
    public function apply(Request $request, Job $job): JsonResponse
    {
        // Check if job is active
        if ($job->status !== 'active') {
            return response()->json([
                'success' => false,
                'message' => 'Job is no longer accepting applications'
            ], 400);
        }

        // Check if user already applied
        $existingApplication = Application::where('user_id', auth()->id())
            ->where('job_id', $job->id)
            ->first();

        if ($existingApplication) {
            return response()->json([
                'success' => false,
                'message' => 'You have already applied for this job'
            ], 400);
        }

        $request->validate([
            'cover_letter' => 'nullable|string|max:2000',
            'expected_salary' => 'nullable|numeric|min:0',
            'availability_date' => 'nullable|date|after:today',
            'additional_info' => 'nullable|string|max:1000',
        ]);

        $user = auth()->user();

        // Check if user has a resume
        if (!$user->profile || !$user->profile->resume_path) {
            return response()->json([
                'success' => false,
                'message' => 'Please upload a resume before applying'
            ], 400);
        }

        $application = Application::create([
            'user_id' => $user->id,
            'job_id' => $job->id,
            'cover_letter' => $request->cover_letter,
            'expected_salary' => $request->expected_salary,
            'availability_date' => $request->availability_date,
            'additional_info' => $request->additional_info,
            'status' => 'pending',
            'applied_at' => now(),
        ]);

        // Send notification to company
        $job->company->notify(new \App\Notifications\ApplicationSubmitted($application));

        return response()->json([
            'success' => true,
            'data' => $application->load(['job.company']),
            'message' => 'Application submitted successfully'
        ], 201);
    }

    /**
     * Get user's applications
     */
    public function myApplications(Request $request): JsonResponse
    {
        $applications = Application::with(['job.company'])
            ->where('user_id', auth()->id())
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $applications,
            'message' => 'Applications retrieved successfully'
        ]);
    }

    /**
     * Get a specific application
     */
    public function show(Application $application): JsonResponse
    {
        // Check if user owns this application
        if ($application->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        $application->load(['job.company', 'user.profile']);

        return response()->json([
            'success' => true,
            'data' => $application,
            'message' => 'Application retrieved successfully'
        ]);
    }

    /**
     * Get company applications (company only)
     */
    public function companyApplications(Request $request): JsonResponse
    {
        $user = User::find(auth()->id());
        $company = $user->company;

        if (!$company) {
            return response()->json([
                'success' => false,
                'message' => 'Company profile required'
            ], 403);
        }

        $applications = Application::with(['user.profile', 'job'])
            ->whereHas('job', function ($query) use ($company) {
                $query->where('company_id', $company->id);
            })
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $applications,
            'message' => 'Company applications retrieved successfully'
        ]);
    }

    /**
     * Update application status (company only)
     */
    public function updateStatus(Request $request, Application $application): JsonResponse
    {
        $user = User::find(auth()->id());
        $company = $user->company;

        if (!$company) {
            return response()->json([
                'success' => false,
                'message' => 'Company profile required'
            ], 403);
        }

        // Check if application belongs to company's job
        if ($application->job->company_id !== $company->id) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        $request->validate([
            'status' => 'required|in:pending,reviewing,shortlisted,interviewed,offered,rejected,withdrawn',
            'notes' => 'nullable|string|max:1000',
        ]);

        $oldStatus = $application->status;

        $application->update([
            'status' => $request->status,
            'review_notes' => $request->notes,
            'reviewed_at' => now(),
            'reviewed_by' => $user->id,
        ]);

        // Send notification to applicant if status changed
        if ($oldStatus !== $request->status) {
            $application->user->notify(new \App\Notifications\ApplicationStatusUpdated($application, $oldStatus));
        }

        return response()->json([
            'success' => true,
            'data' => $application->load(['user.profile', 'job']),
            'message' => 'Application status updated successfully'
        ]);
    }

    /**
     * Download application resume
     */
    public function downloadResume(Application $application): JsonResponse
    {
        $user = User::find(auth()->id());
        $company = $user->company;

        if (!$company) {
            return response()->json([
                'success' => false,
                'message' => 'Company profile required'
            ], 403);
        }

        // Check if application belongs to company's job
        if ($application->job->company_id !== $company->id) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        $userProfile = $application->user->profile;

        if (!$userProfile || !$userProfile->resume_path) {
            return response()->json([
                'success' => false,
                'message' => 'Resume not found'
            ], 404);
        }

        $fileUrl = Storage::disk('public')->url($userProfile->resume_path);

        return response()->json([
            'success' => true,
            'data' => [
                'download_url' => $fileUrl,
                'filename' => basename($userProfile->resume_path)
            ],
            'message' => 'Resume download URL generated'
        ]);
    }
}