<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Job;
use App\Models\JobComment;
use App\Models\JobLike;
use App\Models\JobShare;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class JobSocialController extends Controller
{
    /**
     * Toggle like on a job.
     *
     * @param  \App\Models\Job  $job
     * @return \Illuminate\Http\JsonResponse
     */
    public function toggleLike(Job $job)
    {
        $user = Auth::user();
        $liked = JobLike::where('user_id', $user->id)
            ->where('job_id', $job->id)
            ->exists();

        if ($liked) {
            JobLike::where('user_id', $user->id)
                ->where('job_id', $job->id)
                ->delete();
            $action = 'unliked';
        } else {
            JobLike::create([
                'user_id' => $user->id,
                'job_id' => $job->id,
            ]);
            $action = 'liked';
        }

        return response()->json([
            'message' => "Job successfully {$action}",
            'likes_count' => $job->fresh()->likes()->count(),
            'action' => $action,
        ]);
    }

    /**
     * Add a comment to a job.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Job  $job
     * @return \Illuminate\Http\JsonResponse
     */
    public function addComment(Request $request, Job $job)
    {
        $request->validate([
            'content' => ['required', 'string', 'max:1000'],
        ]);

        $comment = JobComment::create([
            'user_id' => Auth::id(),
            'job_id' => $job->id,
            'content' => $request->input('content'),
            'parent_id' => $request->parent_id,
        ]);

        return response()->json([
            'message' => 'Comment added successfully',
            'comment' => [
                'id' => $comment->id,
                'content' => $comment->content,
                'created_at' => $comment->created_at->diffForHumans(),
                'user' => [
                    'name' => $comment->user->name,
                    'avatar' => $comment->user->avatar_url,
                ],
            ],
            'comments_count' => $job->fresh()->comments()->count(),
        ]);
    }

    /**
     * Get comments for a job.
     *
     * @param  \App\Models\Job  $job
     * @return \Illuminate\Http\JsonResponse
     */
    public function getComments(Job $job)
    {
        $comments = $job->comments()
            ->with('user')
            ->orderByDesc('created_at')
            ->paginate(10);

        return response()->json([
            'comments' => collect($comments->items())->map(function ($comment) {
                return [
                    'id' => $comment->id,
                    'content' => $comment->content,
                    'created_at' => $comment->created_at->diffForHumans(),
                    'user' => [
                        'name' => $comment->user->name,
                        'avatar' => $comment->user->avatar_url,
                    ],
                    'likes_count' => $comment->likes()->count(),
                    'replies_count' => $comment->replies()->count(),
                ];
            }),
            'pagination' => [
                'current_page' => $comments->currentPage(),
                'last_page' => $comments->lastPage(),
                'per_page' => $comments->perPage(),
                'total' => $comments->total(),
            ],
        ]);
    }

    /**
     * Record a job share.
     *
     * @param  \App\Models\Job  $job
     * @return \Illuminate\Http\JsonResponse
     */
    public function recordShare(Job $job)
    {
        JobShare::create([
            'user_id' => Auth::id(),
            'job_id' => $job->id,
            'platform' => request('platform', 'web'),
        ]);

        return response()->json([
            'message' => 'Share recorded successfully',
            'shares_count' => $job->fresh()->shares()->count(),
        ]);
    }

    /**
     * Get a job's social stats.
     *
     * @param  \App\Models\Job  $job
     * @return \Illuminate\Http\JsonResponse
     */
    public function getStats(Job $job)
    {
        $user = Auth::user();

        return response()->json([
            'likes_count' => $job->likes()->count(),
            'comments_count' => $job->comments()->count(),
            'shares_count' => $job->shares()->count(),
            'liked_by_user' => $user ? $job->likes()->where('user_id', $user->id)->exists() : false,
            'engagement_rate' => $this->calculateEngagementRate($job),
        ]);
    }

    /**
     * Calculate engagement rate for a job.
     *
     * @param  \App\Models\Job  $job
     * @return float
     */
    private function calculateEngagementRate(Job $job)
    {
        $totalEngagements = $job->likes()->count() +
            $job->comments()->count() +
            $job->shares()->count();

        $views = $job->views_count ?? 0;

        return $views > 0 ? round(($totalEngagements / $views) * 100, 2) : 0;
    }
}
