<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Services\PaymentService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PaymentController extends Controller
{
    public function __construct(
        private PaymentService $paymentService
    ) {}

    /**
     * Initiate a payment
     */
    public function initiate(Request $request): JsonResponse
    {
        $request->validate([
            'amount' => 'required|numeric|min:100',
            'currency' => 'required|string|size:3',
            'purpose' => 'required|string|max:255',
            'msisdn' => 'required|string|regex:/^[+\d]+$/',
            'company_id' => 'nullable|exists:companies,id',
            'metadata' => 'nullable|array',
        ]);

        try {
            // Record transaction
            $txId = DB::table('payment_transactions')->insertGetId([
                'company_id' => $request->company_id,
                'amount_cfa' => $request->amount,
                'currency' => $request->currency,
                'status' => 'pending',
                'purpose' => $request->purpose,
                'meta' => json_encode($request->metadata ?? []),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Prepare Mesomb payload
            $payload = [
                'amount' => $request->amount,
                'payer' => $request->msisdn,
                'fees' => 0,
                'trxref' => 'HC-' . $txId,
                'service' => config('services.mesomb.service'),
                'country' => config('services.mesomb.country'),
            ];

            $response = $this->paymentService->initiatePayment($payload);

            // Update transaction with provider reference
            DB::table('payment_transactions')
                ->where('id', $txId)
                ->update([
                    'provider_ref' => $response['reference'] ?? null,
                    'meta' => json_encode(array_merge($request->metadata ?? [], [
                        'mesomb_response' => $response
                    ])),
                    'updated_at' => now(),
                ]);

            return response()->json([
                'success' => true,
                'data' => [
                    'transaction_id' => $txId,
                    'reference' => $response['reference'] ?? null,
                    'status' => 'pending',
                ],
                'message' => 'Payment initiated successfully'
            ]);

        } catch (\Throwable $e) {
            Log::error('Payment initiation failed', [
                'error' => $e->getMessage(),
                'request' => $request->all()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to initiate payment'
            ], 500);
        }
    }

    /**
     * Check payment status
     */
    public function status(Request $request, string $transactionId): JsonResponse
    {
        $transaction = DB::table('payment_transactions')
            ->where('id', $transactionId)
            ->first();

        if (!$transaction) {
            return response()->json([
                'success' => false,
                'message' => 'Transaction not found'
            ], 404);
        }

        // Check if user owns this transaction (if company_id is set)
        if ($transaction->company_id && auth()->check()) {
            $user = auth()->user();
            if (!$user->company || $user->company->id !== $transaction->company_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized'
                ], 403);
            }
        }

        return response()->json([
            'success' => true,
            'data' => [
                'transaction_id' => $transaction->id,
                'status' => $transaction->status,
                'amount' => $transaction->amount_cfa,
                'currency' => $transaction->currency,
                'purpose' => $transaction->purpose,
                'created_at' => $transaction->created_at,
                'updated_at' => $transaction->updated_at,
            ],
            'message' => 'Transaction status retrieved successfully'
        ]);
    }

    /**
     * Get user's payment history
     */
    public function history(Request $request): JsonResponse
    {
        $user = auth()->user();
        $company = $user->company;

        if (!$company) {
            return response()->json([
                'success' => false,
                'message' => 'Company profile required'
            ], 403);
        }

        $transactions = DB::table('payment_transactions')
            ->where('company_id', $company->id)
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $transactions,
            'message' => 'Payment history retrieved successfully'
        ]);
    }

    /**
     * Process refund (admin only)
     */
    public function refund(Request $request, string $transactionId): JsonResponse
    {
        $request->validate([
            'amount' => 'nullable|numeric|min:1',
            'reason' => 'required|string|max:500',
        ]);

        // Check admin permission
        if (!auth()->user()->hasRole('admin')) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        $transaction = DB::table('payment_transactions')
            ->where('id', $transactionId)
            ->where('status', 'success')
            ->first();

        if (!$transaction) {
            return response()->json([
                'success' => false,
                'message' => 'Transaction not found or not eligible for refund'
            ], 404);
        }

        $refundAmount = $request->amount ?? $transaction->amount_cfa;

        if ($refundAmount > $transaction->amount_cfa) {
            return response()->json([
                'success' => false,
                'message' => 'Refund amount cannot exceed transaction amount'
            ], 400);
        }

        try {
            // Record refund transaction
            $refundTxId = DB::table('payment_transactions')->insertGetId([
                'company_id' => $transaction->company_id,
                'amount_cfa' => -$refundAmount,
                'currency' => $transaction->currency,
                'status' => 'refunded',
                'purpose' => 'refund',
                'meta' => json_encode([
                    'original_transaction_id' => $transaction->id,
                    'reason' => $request->reason,
                ]),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Update original transaction
            DB::table('payment_transactions')
                ->where('id', $transaction->id)
                ->update([
                    'status' => 'refunded',
                    'meta' => json_encode(array_merge(
                        json_decode($transaction->meta, true) ?? [],
                        ['refunded' => true, 'refund_id' => $refundTxId]
                    )),
                    'updated_at' => now(),
                ]);

            // Handle subscription cancellation if it was a subscription payment
            if (str_contains($transaction->purpose, 'subscription')) {
                DB::table('subscriptions')
                    ->where('company_id', $transaction->company_id)
                    ->where('status', 'active')
                    ->update([
                        'status' => 'cancelled',
                        'cancelled_at' => now(),
                        'updated_at' => now(),
                    ]);
            }

            Log::info('Refund processed', [
                'original_transaction' => $transaction->id,
                'refund_transaction' => $refundTxId,
                'amount' => $refundAmount,
                'reason' => $request->reason,
            ]);

            return response()->json([
                'success' => true,
                'data' => [
                    'refund_transaction_id' => $refundTxId,
                    'amount' => $refundAmount,
                ],
                'message' => 'Refund processed successfully'
            ]);

        } catch (\Throwable $e) {
            Log::error('Refund processing failed', [
                'transaction_id' => $transactionId,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to process refund'
            ], 500);
        }
    }
}
