<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    /**
     * Get user profile
     */
    public function profile(): JsonResponse
    {
        $user = User::with(['profile', 'company'])->find(auth()->id());

        return response()->json([
            'success' => true,
            'data' => $user,
            'message' => 'Profile retrieved successfully'
        ]);
    }

    /**
     * Update user profile
     */
    public function updateProfile(Request $request): JsonResponse
    {
        $user = User::find(auth()->id());

        $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => ['required', 'email', Rule::unique('users')->ignore($user->id)],
            'phone' => 'nullable|string|max:20',
            'date_of_birth' => 'nullable|date|before:today',
            'location' => 'nullable|string|max:255',
            'bio' => 'nullable|string|max:1000',
            'current_job_title' => 'nullable|string|max:255',
            'experience_years' => 'nullable|string',
            'skills' => 'nullable|string|max:1000',
            'education_level' => 'nullable|string',
            'expected_salary' => 'nullable|numeric|min:0',
            'availability' => 'nullable|string',
            'remote_preference' => 'boolean',
            'email_notifications' => 'boolean',
            'preferred_industries' => 'nullable|array',
        ]);

        // Update user basic info
        $user->name = $request->first_name . ' ' . $request->last_name;
        $user->email = $request->email;
        $user->save();

        // Update or create profile
        $profileData = [
            'phone' => $request->phone,
            'date_of_birth' => $request->date_of_birth,
            'location' => $request->location,
            'bio' => $request->bio,
            'current_job_title' => $request->current_job_title,
            'experience_years' => $request->experience_years,
            'skills' => $request->skills,
            'education_level' => $request->education_level,
            'expected_salary' => $request->expected_salary,
            'availability' => $request->availability,
            'remote_preference' => $request->boolean('remote_preference'),
            'email_notifications' => $request->boolean('email_notifications'),
            'preferred_industries' => $request->preferred_industries ? json_encode($request->preferred_industries) : null,
        ];

        $user->profile()->updateOrCreate(
            ['user_id' => $user->id],
            $profileData
        );

        return response()->json([
            'success' => true,
            'data' => User::with('profile')->find($user->id),
            'message' => 'Profile updated successfully'
        ]);
    }

    /**
     * Upload resume
     */
    public function uploadResume(Request $request): JsonResponse
    {
        $request->validate([
            'resume' => 'required|file|mimes:pdf,doc,docx|max:5120',
        ]);

        $user = User::find(auth()->id());

        // Delete old resume if exists
        if ($user->profile && $user->profile->resume_path) {
            Storage::disk('public')->delete($user->profile->resume_path);
        }

        $resumePath = $request->file('resume')->store('resumes', 'public');

        $user->profile()->updateOrCreate(
            ['user_id' => $user->id],
            ['resume_path' => $resumePath]
        );

        return response()->json([
            'success' => true,
            'data' => [
                'resume_path' => $resumePath,
                'filename' => $request->file('resume')->getClientOriginalName()
            ],
            'message' => 'Resume uploaded successfully'
        ]);
    }

    /**
     * Update password
     */
    public function updatePassword(Request $request): JsonResponse
    {
        $request->validate([
            'current_password' => 'required',
            'password' => 'required|min:8|confirmed',
        ]);

        $user = User::find(auth()->id());

        // Check current password
        if (!Hash::check($request->current_password, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Current password is incorrect'
            ], 400);
        }

        $user->update([
            'password' => Hash::make($request->password),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Password updated successfully'
        ]);
    }

    /**
     * Delete account
     */
    public function deleteAccount(Request $request): JsonResponse
    {
        $request->validate([
            'password' => 'required',
        ]);

        $user = User::find(auth()->id());

        // Check password
        if (!Hash::check($request->password, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Password is incorrect'
            ], 400);
        }

        // Delete profile and related data
        if ($user->profile) {
            if ($user->profile->resume_path) {
                Storage::disk('public')->delete($user->profile->resume_path);
            }
            $user->profile->delete();
        }

        // Delete applications
        $user->applications()->delete();

        // Delete user
        $user->delete();

        return response()->json([
            'success' => true,
            'message' => 'Account deleted successfully'
        ]);
    }
}