<?php

namespace App\Http\Controllers;

use App\Models\Application;
use App\Models\Job;
use App\Services\NotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Mail;
use App\Mail\ApplicationStatusUpdated;

class ApplicationController extends Controller
{
    protected NotificationService $notificationService;

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }
    /**
     * Store a new job application.
     */
    public function store(Request $request)
    {
        $request->validate([
            'job_id' => 'required|exists:jobs,id',
            'cover_letter' => 'nullable|string|max:2000',
            'expected_salary' => 'nullable|numeric|min:0',
            'availability_date' => 'nullable|date|after:today',
            'resume' => 'nullable|file|mimes:pdf,doc,docx|max:5120', // 5MB max
        ]);

        $job = Job::findOrFail($request->job_id);

        // Check if user already applied
        $existingApplication = Application::where('job_id', $job->id)
            ->where('user_id', auth()->id())
            ->first();

        if ($existingApplication) {
            return back()->with('error', 'You have already applied for this position.');
        }

        // Handle resume upload
        $resumePath = null;
        if ($request->hasFile('resume')) {
            $file = $request->file('resume');
            $originalName = $file->getClientOriginalName();
            $extension = $file->getClientOriginalExtension();
            $filename = time() . '_' . auth()->id() . '_' . $job->id . '.' . $extension;
            $resumePath = $file->storeAs('resumes', $filename, 'public');
        }

        // Create application
        $application = Application::create([
            'job_id' => $job->id,
            'user_id' => auth()->id(),
            'cover_letter' => $request->cover_letter,
            'expected_salary' => $request->expected_salary,
            'availability_date' => $request->availability_date,
            'resume_path' => $resumePath,
            'status' => 'pending',
        ]);

        // Send confirmation notification
        $this->notificationService->send(
            auth()->user(),
            'in_app',
            'Application Submitted',
            "Your application for {$job->title} at {$job->company->name} has been submitted successfully.",
            [],
            null,
            $application
        );

        return back()->with('success', 'Your application has been submitted successfully!');
    }

    /**
     * Display user's applications.
     */
    public function index()
    {
        $applications = Application::with(['job.company'])
            ->where('user_id', auth()->id())
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('applications.index', compact('applications'));
    }

    /**
     * Show application details.
     */
    public function show(Application $application)
    {
        // Ensure user owns this application
        if ($application->user_id !== auth()->id()) {
            abort(403);
        }

        $application->load(['job.company']);

        return view('applications.show', compact('application'));
    }

    /**
     * Withdraw an application.
     */
    public function withdraw(Application $application)
    {
        // Ensure user owns this application
        if ($application->user_id !== auth()->id()) {
            abort(403);
        }

        // Only allow withdrawal if application is still pending
        if ($application->status !== 'pending') {
            return back()->with('error', 'You cannot withdraw an application that has already been reviewed.');
        }

        $application->update(['status' => 'withdrawn']);

        return back()->with('success', 'Your application has been withdrawn.');
    }

    /**
     * Download resume.
     */
    public function downloadResume(Application $application)
    {
        // Ensure user owns this application or is the job poster
        if ($application->user_id !== auth()->id() && $application->job->company->owner_user_id !== auth()->id()) {
            abort(403);
        }

        if (!$application->resume_path) {
            abort(404);
        }

        return Storage::disk('public')->download($application->resume_path);
    }

    /**
     * Show applications for a specific job (for employers).
     */
    public function jobApplications(Job $job)
    {
        // Ensure user owns the company that posted this job
        if ($job->company->owner_user_id !== auth()->id()) {
            abort(403);
        }

        $applications = Application::with('user')
            ->where('job_id', $job->id)
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('applications.job-applications', compact('job', 'applications'));
    }

    /**
     * Update application status (for employers).
     */
    public function updateStatus(Request $request, Application $application)
    {
        // Ensure employer owns the job
        if ($application->job->company->owner_user_id !== auth()->id()) {
            abort(403);
        }

        $request->validate([
            'status' => 'required|in:pending,reviewing,shortlisted,interviewed,offered,rejected',
            'notes' => 'nullable|string|max:1000',
        ]);

        $application->update([
            'status' => $request->status,
            'employer_notes' => $request->notes,
        ]);

        // Send email notification to applicant
        Mail::to($application->user->email)->send(new ApplicationStatusUpdated($application));

        // Send in-app notification
        $this->notificationService->send(
            $application->user,
            'in_app',
            'Application Status Updated',
            "Your application status for {$application->job->title} has been updated to: " . ucfirst($request->status),
            [],
            auth()->user(),
            $application
        );

        return back()->with('success', 'Application status updated successfully.');
    }
}