<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Organization;
use App\Providers\RouteServiceProvider;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use Illuminate\View\View;

class RegisteredUserController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create(): View
    {
        return view('auth.register');
    }

    /**
     * Handle personal account registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function storePersonal(Request $request): RedirectResponse
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:'.User::class],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'terms' => ['required', 'accepted'],
        ]);

        try {
            DB::beginTransaction();

            // Create user as personal
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'account_type' => 'personal',
            ]);

            DB::commit();

            event(new Registered($user));
            Auth::login($user);

            return redirect(RouteServiceProvider::HOME)->with('success',
                'Welcome! Your personal account has been created successfully. You can now start applying for jobs.'
            );

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Registration failed. Please try again.'])->withInput();
        }
    }

    /**
     * Handle company account registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function storeCompany(Request $request): RedirectResponse
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:'.User::class],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'position' => ['required', 'string', 'max:255'],
            'company_name' => ['required', 'string', 'max:255'],
            'company_sector' => ['required', 'string', 'max:255'],
            'company_size' => ['required', 'string', 'max:50'],
            'company_website' => ['nullable', 'url', 'max:255'],
            'company_social' => ['nullable', 'url', 'max:255'],
            'terms' => ['required', 'accepted'],
        ]);

        try {
            DB::beginTransaction();

            // Create user as company
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'account_type' => 'company',
                'position' => $request->position,
            ]);

            // Create company profile
            Organization::create([
                'name' => $request->company_name,
                'sector' => $request->company_sector,
                'company_size' => $request->company_size,
                'website' => $request->company_website,
                'owner_user_id' => $user->id,
            ]);

            DB::commit();

            event(new Registered($user));
            Auth::login($user);

            return redirect()->route('dashboard.company')->with('success',
                'Welcome! Your company account has been created successfully. You can now post jobs and start hiring.'
            );

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Registration failed. Please try again.'])->withInput();
        }
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        $validationRules = [
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:'.User::class],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'account_type' => ['required', 'in:personal,company'],
            'terms' => ['required', 'accepted'],
        ];

        // Add company-specific validation rules
        if ($request->account_type === 'company') {
            $validationRules = array_merge($validationRules, [
                'company_name' => ['required', 'string', 'max:255'],
                'position' => ['required', 'string', 'max:255'],
                'company_sector' => ['nullable', 'string', 'max:255'],
                'company_size' => ['nullable', 'string', 'max:50'],
                'company_registration_no' => ['nullable', 'string', 'max:255'],
                'company_website' => ['nullable', 'url', 'max:255'],
                'company_description' => ['nullable', 'string', 'max:1000'],
            ]);
        }

        $request->validate($validationRules);

        try {
            DB::beginTransaction();

            // Create user
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'account_type' => $request->account_type,
                'position' => $request->position,
            ]);

            // Create company profile if user registered as company
            if ($user->isCompany()) {
                Organization::create([
                    'name' => $request->company_name,
                    'description' => $request->company_description,
                    'sector' => $request->company_sector,
                    'company_size' => $request->company_size,
                    'registration_no' => $request->company_registration_no,
                    'website' => $request->company_website,
                    'owner_user_id' => $user->id,
                ]);
            }

            DB::commit();

            event(new Registered($user));
            Auth::login($user);

            // Redirect based on account type
            if ($user->isCompany()) {
                return redirect()->route('dashboard.company')->with('success',
                    'Welcome! Your company account has been created successfully. You can now post jobs and start hiring.'
                );
            }

            return redirect(RouteServiceProvider::HOME)->with('success',
                'Welcome! Your account has been created successfully.'
            );

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Registration failed. Please try again.']);
        }
    }
}
