<?php

namespace App\Http\Controllers;

use App\Models\Organization;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Database\Eloquent\Builder;
use Stancl\Tenancy\Database\Models\Tenant;

class CompanyController extends Controller
{
    public function create()
    {
        return view('companies.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'sector' => 'nullable|string|max:255',
            'registration_no' => 'nullable|string|max:255',
            'domain' => 'nullable|string|max:255',
        ]);

        $data['owner_user_id'] = Auth::id();

        $company = Organization::create($data);

        // Optional tenant domain provisioning
        if (!empty($data['domain'])) {
            $tenantId = 'company_'.$company->id;
            $tenant = Tenant::firstOrCreate([
                'id' => $tenantId,
            ], [
                'data' => [
                    'company_id' => $company->id,
                    'name' => $company->name,
                ],
            ]);
            if (!$tenant->domains()->where('domain', $data['domain'])->exists()) {
                $tenant->domains()->create(['domain' => $data['domain']]);
            }
        }

        return redirect()->route('dashboard')->with('status', 'Company created.');
    }

    public function showCompany(Organization $company)
    {
        // Check if user owns this company
        if ($company->owner_user_id !== Auth::id()) {
            abort(403, 'Unauthorized');
        }

        $company->load(['jobs' => function ($query) {
            $query->orderBy('created_at', 'desc')->limit(10);
        }, 'subscription']);

        return view('companies.show', compact('company'));
    }

    public function edit(Organization $company)
    {
        // Check if user owns this company
        if ($company->owner_user_id !== Auth::id()) {
            abort(403, 'Unauthorized');
        }

        return view('companies.edit', compact('company'));
    }

    public function update(Request $request, Organization $company)
    {
        // Check if user owns this company
        if ($company->owner_user_id !== Auth::id()) {
            abort(403, 'Unauthorized');
        }

        $data = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'sector' => 'nullable|string|max:255',
            'registration_no' => 'nullable|string|max:255',
            'domain' => 'nullable|string|max:255',
            'location' => 'nullable|string|max:255',
            'website' => 'nullable|url',
            'size' => 'nullable|string|in:1-10,11-50,51-200,201-500,501-1000,1000+',
            'founded_year' => 'nullable|integer|min:1800|max:' . (date('Y') + 1),
        ]);

        $company->update($data);

        return redirect()->route('companies.show', $company)->with('status', 'Company updated successfully.');
    }

    public function index(Request $request)
    {
        $query = Organization::with(['owner', 'jobs' => function ($query) {
            $query->where('status', 'active');
        }])
        ->withCount(['jobs' => function ($query) {
            $query->where('status', 'active');
        }]);

        // Search functionality
        if ($request->filled('q')) {
            $searchTerm = $request->q;
            $query->where(function (Builder $q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('description', 'like', "%{$searchTerm}%")
                  ->orWhere('sector', 'like', "%{$searchTerm}%");
            });
        }

        // Location filter
        if ($request->filled('location')) {
            $query->where('location', 'like', "%{$request->location}%");
        }

        // Sector filter
        if ($request->filled('sector')) {
            $query->where('sector', $request->sector);
        }

        // Size filter
        if ($request->filled('size')) {
            $query->where('size', $request->size);
        }

        $companies = $query->orderBy('created_at', 'desc')->paginate(12);

        return view('companies.index', compact('companies'));
    }

    public function show(Organization $company)
    {
        $company->load(['jobs' => function ($query) {
            $query->where('status', 'active');

            if (\Illuminate\Support\Facades\Schema::hasColumn('jobs', 'application_deadline')) {
                $query->where(function ($q) {
                    $q->where('application_deadline', '>', now())
                      ->orWhereNull('application_deadline');
                });
            }

            $query->orderBy('created_at', 'desc');
        }, 'owner']);

        return view('companies.show', compact('company'));
    }

    public function destroy(Organization $company)
    {
        // Check if user owns this company
        if ($company->owner_user_id !== Auth::id()) {
            abort(403, 'Unauthorized');
        }

        // Check if company has active jobs
        if ($company->jobs()->where('status', 'active')->exists()) {
            return redirect()->back()->withErrors(['error' => 'Cannot delete company with active job postings.']);
        }

        $company->delete();

        return redirect()->route('dashboard')->with('status', 'Company deleted successfully.');
    }
}


