<?php

namespace App\Http\Controllers;

use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use App\Models\User;

class InstallController extends Controller
{
    public function index()
    {
        $isInstalled = file_exists(storage_path('app/hirecameroon_installed')) || env('APP_INSTALLED', false);
        if ($isInstalled) {
            return redirect('/');
        }
        return view('install.index');
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'app_url' => 'nullable|url',
            'db_host' => 'required|string',
            'db_port' => 'required|numeric',
            'db_database' => 'required|string',
            'db_username' => 'required|string',
            'db_password' => 'nullable|string',
            'admin_name' => 'required|string',
            'admin_email' => 'required|email',
            'admin_password' => 'required|string|min:8',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        // Write env values
        $this->writeEnv([
            'APP_NAME' => 'HireCameroon',
            'APP_URL' => $request->input('app_url', 'http://localhost'),
            'APP_INSTALLED' => 'true',
            'DB_CONNECTION' => 'mysql',
            'DB_HOST' => $request->input('db_host'),
            'DB_PORT' => $request->input('db_port'),
            'DB_DATABASE' => $request->input('db_database'),
            'DB_USERNAME' => $request->input('db_username'),
            'DB_PASSWORD' => $request->input('db_password'),
            'APP_LOCALE' => 'en',
            'APP_FALLBACK_LOCALE' => 'en',
        ]);

        // Run key generate
        Artisan::call('key:generate', ['--force' => true]);

        // Prefer SQL import if provided
        $sqlPath = database_path('HireCameroon.sql');
        if (file_exists($sqlPath)) {
            // Basic import; assumes valid SQL dump in correct order
            DB::statement('SET FOREIGN_KEY_CHECKS=0');
            DB::unprepared(file_get_contents($sqlPath));
            DB::statement('SET FOREIGN_KEY_CHECKS=1');
        } else {
            // Fall back to migrations + seeding
            Artisan::call('migrate', ['--force' => true]);
            Artisan::call('db:seed', ['--force' => true]);
        }

        // Create admin user if not exists
        $admin = User::firstOrCreate(
            ['email' => $request->input('admin_email')],
            [
                'name' => $request->input('admin_name'),
                'password' => Hash::make($request->input('admin_password')),
                'role' => 'admin',
                'preferred_language' => 'en',
            ]
        );

        // Mark installed
        File::put(storage_path('app/hirecameroon_installed'), now()->toDateTimeString());

        return redirect()->route('install.success');
    }

    public function success()
    {
        return view('install.success');
    }

    private function writeEnv(array $pairs): void
    {
        $envPath = base_path('.env');
        $content = File::exists($envPath) ? File::get($envPath) : '';

        foreach ($pairs as $key => $value) {
            $pattern = "/^{$key}=.*$/m";
            $line = $key.'='.str_replace(['\n', '"'], ['\\n', ''], (string) $value);
            if (preg_match($pattern, $content)) {
                $content = preg_replace($pattern, $line, $content);
            } else {
                $content .= (strlen($content) ? "\n" : '').$line;
            }
        }

        File::put($envPath, $content);
    }
}


