<?php

namespace App\Http\Controllers;

use App\Models\Job;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\Builder;

class JobController extends Controller
{
    /**
     * Display a listing of jobs with search and filtering.
     */
    public function index(Request $request)
    {
        $query = Job::with(['company'])
            ->withCount('applications')
            ->where('status', 'active');

        // Only apply application_deadline filter if column exists (prevents SQL errors during migrations)
        if (\Illuminate\Support\Facades\Schema::hasColumn('jobs', 'application_deadline')) {
            $query->where(function ($q) {
                $q->where('application_deadline', '>', now())
                  ->orWhereNull('application_deadline');
            });
        }

        // Advanced Search functionality
        if ($request->filled('q')) {
            $searchTerm = $request->q;
            $query->where(function (Builder $q) use ($searchTerm) {
                $q->where('title', 'like', "%{$searchTerm}%")
                  ->orWhere('description', 'like', "%{$searchTerm}%")
                  ->orWhere('requirements', 'like', "%{$searchTerm}%")
                  ->orWhere('tags', 'like', "%{$searchTerm}%")
                  ->orWhereHas('company', function (Builder $q) use ($searchTerm) {
                      $q->where('name', 'like', "%{$searchTerm}%")
                        ->orWhere('description', 'like', "%{$searchTerm}%");
                  });
            });
        }

        // Location filter
        if ($request->filled('location')) {
            if ($request->location === 'remote') {
                // No explicit column in schema; treat as city/region containing 'Remote'
                $query->where(function ($q) {
                    $q->where('location_city', 'like', '%remote%')
                      ->orWhere('location_region', 'like', '%remote%');
                });
            } else {
                $loc = $request->location;
                $query->where(function ($q) use ($loc) {
                    $q->where('location_city', 'like', "%{$loc}%")
                      ->orWhere('location_region', 'like', "%{$loc}%");
                });
            }
        }

        // Sector/Category filter
        if ($request->filled('sector')) {
            $query->where('sector_type', $request->sector);
        }

        // Experience level filter
        if ($request->filled('experience_level')) {
            $query->where('experience_level', $request->experience_level);
        }

        // Employment type filter
        if ($request->filled('employment_type')) {
            $query->where('job_type', $request->employment_type);
        }

        // Salary range filters
        if ($request->filled('salary_min')) {
            $query->where('salary_max', '>=', $request->salary_min);
        }
        if ($request->filled('salary_max')) {
            $query->where('salary_min', '<=', $request->salary_max);
        }

        // Posted within filter
        if ($request->filled('posted_within')) {
            $days = (int) $request->posted_within;
            $query->where('created_at', '>=', now()->subDays($days));
        }

        // Remote jobs filter
        // Remote flag fallback: treat as textual hint in location
        if ($request->boolean('remote')) {
            $query->where(function ($q) {
                $q->where('location_city', 'like', '%remote%')
                  ->orWhere('location_region', 'like', '%remote%');
            });
        }

        // Featured jobs filter
        if ($request->boolean('featured')) {
            $query->where('featured', true);
        }

        // Jobs with salary info filter
        if ($request->boolean('with_salary')) {
            $query->whereNotNull('salary_min');
        }

        // Advanced Sorting
        $sortBy = $request->get('sort', 'newest');
        switch ($sortBy) {
            case 'oldest':
                $query->orderBy('created_at', 'asc');
                break;
            case 'salary_high':
                $query->orderByRaw('COALESCE(salary_max, salary_min) DESC');
                break;
            case 'salary_low':
                $query->orderByRaw('COALESCE(salary_min, salary_max) ASC');
                break;
            case 'relevance':
                // If there's a search term, prioritize relevance
                if ($request->filled('q')) {
                    $searchTerm = $request->q;
                    $query->orderByRaw("
                        CASE
                            WHEN title LIKE ? THEN 1
                            WHEN description LIKE ? THEN 2
                            WHEN skills LIKE ? THEN 3
                            ELSE 4
                        END
                    ", ["%{$searchTerm}%", "%{$searchTerm}%", "%{$searchTerm}%"])
                    ->orderBy('created_at', 'desc');
                } else {
                    $query->orderBy('created_at', 'desc');
                }
                break;
            case 'newest':
            default:
                $query->orderBy('created_at', 'desc');
                break;
        }

        $jobs = $query->paginate(12)->appends($request->query());

        return view('jobs.index', compact('jobs'));
    }

    /**
     * Display the specified job.
     */
    public function show(Job $job)
    {
        // Ensure job is active and not expired
        if ($job->status !== 'active' || ($job->application_deadline && $job->application_deadline->isPast())) {
            abort(404);
        }

        // Increment view count (you might want to add this to the Job model)
        // $job->increment('views_count');

        // Get similar jobs (basic implementation)
        $similarJobs = Job::where('id', '!=', $job->id)
            ->where('status', 'active')
            ->where(function ($query) use ($job) {
                $query->where('sector_type', $job->sector_type)
                      ->orWhere('location_city', 'like', '%' . $job->location_city . '%')
                      ->orWhere('location_region', 'like', '%' . $job->location_region . '%');
            })
            ->limit(3)
            ->get();

        return view('jobs.show', compact('job', 'similarJobs'));
    }

    /**
     * Show the form for creating a new job.
     */
    public function create()
    {
        // Check if user can post jobs
        if (!auth()->user()->company && !auth()->user()->is_admin) {
            return redirect()->route('companies.create')->with('error', 'Please create your company profile first.');
        }

        return view('jobs.create');
    }

    /**
     * Store a newly created job.
     */
    public function store(Request $request)
    {
        // Check if user can post jobs
        if (!auth()->user()->company && !auth()->user()->is_admin) {
            return redirect()->route('companies.create')->with('error', 'Please create your company profile first.');
        }

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'location_city' => 'nullable|string|max:255',
            'location_region' => 'nullable|string|max:255',
            'job_type' => 'required|in:full-time,part-time,contract,freelance,internship',
            'experience_level' => 'required|in:entry,mid,senior,executive',
            'salary_min' => 'nullable|numeric|min:0',
            'salary_max' => 'nullable|numeric|min:0|gte:salary_min',
            'sector_type' => 'nullable|string|max:255',
            'tags' => 'nullable|array',
            'requirements' => 'nullable|string',
            'application_deadline' => 'nullable|date|after:today',
            'featured' => 'boolean',
        ]);

        // Get the authenticated user's company
        $user = auth()->user();

        // Ensure user has a company profile
        if (!$user->company) {
            return redirect()->route('companies.create')->with('error', 'Please create your company profile first.');
        }

        $validated['company_id'] = $user->company->id;
        $validated['status'] = 'active';

        $job = Job::create($validated);

        return redirect()->route('jobs.index')->with('success', 'Job posted successfully!');
    }
}