<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\Subscription;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Services\PaymentService;

class SubscribeController extends Controller
{
    public function show(Company $company)
    {
        // Check if user owns this company
        if ($company->owner_user_id !== auth()->id()) {
            abort(403, 'Unauthorized');
        }

        $subscription = $company->subscription;
        $plans = [
            'basic' => [
                'name' => 'Basic',
                'price' => 5000,
                'features' => ['5 job posts', 'Basic support']
            ],
            'pro' => [
                'name' => 'Pro',
                'price' => 15000,
                'features' => ['50 job posts', 'ATS features', 'Priority support', 'Analytics']
            ],
            'enterprise' => [
                'name' => 'Enterprise',
                'price' => 50000,
                'features' => ['Unlimited job posts', 'Advanced ATS', 'Dedicated support', 'Custom features']
            ]
        ];

        return view('subscriptions.subscribe', compact('company', 'subscription', 'plans'));
    }

    public function start(Request $request, Company $company, PaymentService $payments)
    {
        // Check if user owns this company
        if ($company->owner_user_id !== auth()->id()) {
            abort(403, 'Unauthorized');
        }

        $request->validate([
            'plan' => 'required|in:basic,pro,enterprise',
            'msisdn' => 'required|string|regex:/^[+\d]+$/',
        ]);

        $plans = [
            'basic' => 5000,
            'pro' => 15000,
            'enterprise' => 50000,
        ];

        $amount = $plans[$request->plan];

        // Record pending transaction
        $txId = DB::table('payment_transactions')->insertGetId([
            'company_id' => $company->id,
            'amount_cfa' => $amount,
            'currency' => 'XAF',
            'status' => 'pending',
            'purpose' => $request->plan . '_subscription',
            'meta' => json_encode(['plan' => $request->plan]),
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // Minimal Mesomb collect payload (adjust per official SDK/requirements)
        $payload = [
            'amount' => $amount,
            'payer' => $request->input('msisdn'),
            'fees' => 0,
            'trxref' => 'HC-'.$txId,
            'service' => config('services.mesomb.service'),
            'country' => config('services.mesomb.country'),
        ];

        try {
            $response = $payments->initiatePayment($payload);
            DB::table('payment_transactions')->where('id', $txId)->update([
                'provider_ref' => $response['reference'] ?? null,
                'meta' => json_encode(['plan' => $request->plan, 'response' => $response]),
                'updated_at' => now(),
            ]);
        } catch (\Throwable $e) {
            Log::error('Mesomb initiation failed', ['error' => $e->getMessage()]);
            return back()->withErrors(['payment' => 'Failed to initiate payment.']);
        }

        return redirect()->route('subscriptions.subscribe', $company)->with('status', 'Payment initiated. Complete on your phone.');
    }

    public function status(Company $company)
    {
        // Check if user owns this company
        if ($company->owner_user_id !== auth()->id()) {
            abort(403, 'Unauthorized');
        }

        $subscription = $company->subscription;
        $transactions = DB::table('payment_transactions')
            ->where('company_id', $company->id)
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return view('subscriptions.status', compact('company', 'subscription', 'transactions'));
    }

    public function cancel(Company $company)
    {
        // Check if user owns this company
        if ($company->owner_user_id !== auth()->id()) {
            abort(403, 'Unauthorized');
        }

        $subscription = $company->subscription;

        if (!$subscription || $subscription->status !== 'active') {
            return redirect()->back()->withErrors(['error' => 'No active subscription to cancel.']);
        }

        // Cancel subscription - it will remain active until end date
        $subscription->update([
            'status' => 'cancelled',
            'cancelled_at' => now(),
        ]);

        return redirect()->route('subscriptions.status', $company)->with('status', 'Subscription cancelled. It will remain active until the end date.');
    }

    public function renew(Request $request, Company $company, PaymentService $payments)
    {
        // Check if user owns this company
        if ($company->owner_user_id !== auth()->id()) {
            abort(403, 'Unauthorized');
        }

        $request->validate([
            'msisdn' => 'required|string|regex:/^[+\d]+$/',
        ]);

        $subscription = $company->subscription;

        if (!$subscription) {
            return redirect()->route('subscriptions.subscribe', $company)->withErrors(['error' => 'No subscription found. Please subscribe first.']);
        }

        // Calculate renewal amount based on plan
        $plans = [
            'Basic' => 5000,
            'Pro' => 15000,
            'Enterprise' => 50000,
        ];

        $amount = $plans[$subscription->plan_name] ?? 15000;

        // Record pending transaction
        $txId = DB::table('payment_transactions')->insertGetId([
            'company_id' => $company->id,
            'amount_cfa' => $amount,
            'currency' => 'XAF',
            'status' => 'pending',
            'purpose' => 'subscription_renewal',
            'meta' => json_encode(['plan' => $subscription->plan_name, 'renewal' => true]),
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // Minimal Mesomb collect payload
        $payload = [
            'amount' => $amount,
            'payer' => $request->input('msisdn'),
            'fees' => 0,
            'trxref' => 'HC-'.$txId,
            'service' => config('services.mesomb.service'),
            'country' => config('services.mesomb.country'),
        ];

        try {
            $response = $payments->initiatePayment($payload);
            DB::table('payment_transactions')->where('id', $txId)->update([
                'provider_ref' => $response['reference'] ?? null,
                'meta' => json_encode(['plan' => $subscription->plan_name, 'renewal' => true, 'response' => $response]),
                'updated_at' => now(),
            ]);
        } catch (\Throwable $e) {
            Log::error('Mesomb renewal initiation failed', ['error' => $e->getMessage()]);
            return back()->withErrors(['payment' => 'Failed to initiate renewal payment.']);
        }

        return redirect()->route('subscriptions.status', $company)->with('status', 'Renewal payment initiated. Complete on your phone.');
    }
}


