<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Notifications\AdExpired;

class ExpireAdJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $adId;

    public function __construct(int $adId)
    {
        $this->adId = $adId;
    }

    public function handle(): void
    {
        try {
            $ad = DB::table('ads')->where('id', $this->adId)->first();
            if (!$ad) {
                Log::warning('ExpireAdJob: ad not found', ['ad_id' => $this->adId]);
                return;
            }

            // mark as expired
            DB::table('ads')->where('id', $this->adId)->update([
                'status' => 'expired',
                'updated_at' => now(),
            ]);

            // notify owner
            $user = DB::table('users')->where('id', $ad->user_id)->first();
            if ($user) {
                try {
                    if (config('database.notifications_table', 'notifications')) {
                        DB::table('notifications')->insert([
                            'id' => (string) \Illuminate\Support\Str::uuid(),
                            'type' => AdExpired::class,
                            'notifiable_type' => 'App\\Models\\User',
                            'notifiable_id' => $user->id,
                            'data' => json_encode(['ad_id' => $ad->id, 'message' => 'Your ad has expired']),
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]);
                    }

                    if (class_exists('App\\Models\\User')) {
                        $userModel = (\App\Models\User::find($user->id));
                        if ($userModel) {
                            $userModel->notify(new AdExpired($ad));
                        }
                    }
                } catch (\Exception $e) {
                    Log::error('Failed to send AdExpired notification', ['error' => $e->getMessage(), 'ad_id' => $ad->id]);
                }
            }

            Log::info('ExpireAdJob completed', ['ad_id' => $this->adId]);
        } catch (\Exception $e) {
            Log::error('ExpireAdJob failed', ['error' => $e->getMessage(), 'ad_id' => $this->adId]);
            throw $e;
        }
    }
}
