<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Ad extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'company_id',
        'ad_plan_id',
        'type',
        'position',
        'title',
        'body_html',
        'image_path',
        'video_url',
        'poster_path',
        'target_url',
        'start_at',
        'end_at',
        'status',
        'meta',
        'impressions_limit',
        'clicks_limit',
    ];

    protected $casts = [
        'start_at' => 'datetime',
        'end_at' => 'datetime',
        'meta' => 'array',
        'impressions_limit' => 'integer',
        'clicks_limit' => 'integer',
    ];

    /**
     * Validation rules for ad creation/update
     */
    public static function rules(string $type = null): array
    {
        $rules = [
            'title' => 'required|string|max:255',
            'target_url' => 'required|url|max:2048',
            'type' => 'required|in:image,video,text',
            'position' => 'required|string|max:50',
            'body_html' => 'nullable|string|max:5000',
            'start_at' => 'nullable|date|after:now',
            'end_at' => 'nullable|date|after:start_at',
        ];

        // Add type-specific validation
        if ($type === 'image' || $type === null) {
            $rules['image_path'] = 'required_if:type,image|nullable|string|max:2048';
        }

        if ($type === 'video' || $type === null) {
            $rules['video_url'] = 'required_if:type,video|nullable|url|max:2048';
            $rules['poster_path'] = 'required_if:type,video|nullable|string|max:2048';
        }

        return $rules;
    }

    /**
     * Get the video service type (youtube, vimeo, etc)
     */
    public function getVideoServiceAttribute(): ?string
    {
        if (!$this->video_url) {
            return null;
        }

        if (str_contains($this->video_url, 'youtube.com') || str_contains($this->video_url, 'youtu.be')) {
            return 'youtube';
        }

        if (str_contains($this->video_url, 'vimeo.com')) {
            return 'vimeo';
        }

        return 'other';
    }

    /**
     * Get the video ID from the URL
     */
    public function getVideoIdAttribute(): ?string
    {
        if (!$this->video_url) {
            return null;
        }

        $service = $this->video_service;

        if ($service === 'youtube') {
            // Handle both youtu.be and youtube.com URLs
            if (str_contains($this->video_url, 'youtu.be/')) {
                return substr(parse_url($this->video_url, PHP_URL_PATH), 1);
            }
            parse_str(parse_url($this->video_url, PHP_URL_QUERY), $params);
            return $params['v'] ?? null;
        }

        if ($service === 'vimeo') {
            return substr(parse_url($this->video_url, PHP_URL_PATH), 1);
        }

        return null;
    }

    public function plan()
    {
        return $this->belongsTo(AdPlan::class, 'ad_plan_id');
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function transactions()
    {
        return $this->hasMany(AdTransaction::class);
    }

    public function impressions()
    {
        return $this->hasMany(AdImpression::class);
    }

    public function clicks()
    {
        return $this->hasMany(AdClick::class);
    }

    public function scopeActive($query)
    {
        return $query->where('status', 'active')
            ->where(function ($q) {
                $q->whereNull('start_at')->orWhere('start_at', '<=', now());
            })
            ->where(function ($q) {
                $q->whereNull('end_at')->orWhere('end_at', '>=', now());
            });
    }
}
