<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class Assessment extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'skill_id',
        'time_limit',
        'questions_count',
    ];

    protected $casts = [
        'time_limit' => 'integer',
        'questions_count' => 'integer',
    ];

    /**
     * Boot the model.
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($assessment) {
            if (empty($assessment->slug)) {
                $assessment->slug = Str::slug($assessment->name);
            }
        });
    }

    /**
     * Get the skill associated with this assessment.
     */
    public function skill(): BelongsTo
    {
        return $this->belongsTo(Skill::class);
    }

    /**
     * Get the results for this assessment.
     */
    public function results(): HasMany
    {
        return $this->hasMany(AssessmentResult::class);
    }

    /**
     * Scope to filter by skill.
     */
    public function scopeBySkill($query, $skillId)
    {
        return $query->where('skill_id', $skillId);
    }

    /**
     * Scope to filter by slug.
     */
    public function scopeBySlug($query, $slug)
    {
        return $query->where('slug', $slug);
    }
}