<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class EmailCampaign extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'subject',
        'content',
        'template_id',
        'recipient_groups',
        'recipient_count',
        'status',
        'scheduled_at',
        'sent_at',
        'emails_sent',
        'emails_opened',
        'emails_clicked',
        'created_by',
        'metadata',
    ];

    protected $casts = [
        'recipient_groups' => 'array',
        'scheduled_at' => 'datetime',
        'sent_at' => 'datetime',
        'metadata' => 'array',
    ];

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function template(): BelongsTo
    {
        return $this->belongsTo(EmailTemplate::class, 'template_id');
    }

    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    public function scopeScheduled($query)
    {
        return $query->where('status', 'scheduled');
    }

    public function scopeSent($query)
    {
        return $query->where('status', 'sent');
    }

    public function getOpenRateAttribute()
    {
        if ($this->emails_sent == 0) return 0;
        return round(($this->emails_opened / $this->emails_sent) * 100, 1);
    }

    public function getClickRateAttribute()
    {
        if ($this->emails_sent == 0) return 0;
        return round(($this->emails_clicked / $this->emails_sent) * 100, 1);
    }

    public function getStatusDisplayAttribute()
    {
        return match($this->status) {
            'draft' => 'Draft',
            'scheduled' => 'Scheduled',
            'sending' => 'Sending',
            'sent' => 'Sent',
            'failed' => 'Failed',
            default => ucfirst($this->status),
        };
    }

    public function getRecipientGroupsDisplayAttribute()
    {
        if (empty($this->recipient_groups)) return 'None';

        $groups = [
            'all_users' => 'All Users',
            'job_seekers' => 'Job Seekers',
            'employers' => 'Employers',
            'inactive_users' => 'Inactive Users',
        ];

        return collect($this->recipient_groups)->map(function($group) use ($groups) {
            return $groups[$group] ?? ucfirst(str_replace('_', ' ', $group));
        })->join(', ');
    }
}