<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Job extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'jobs_optimized';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'organization_id',
        'title',
        'slug',
        'description',
        'location_region',
        'location_city',
        'employment_type',
        'experience_level',
        'salary_min',
        'salary_max',
        'currency',
        'negotiable',
        'application_deadline',
        'featured_until',
        'requirements',
        'benefits',
        'application_url',
        'verified_company',
        'scam_flags',
        'verification_status',
        'status',
        'seo_data',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'application_deadline' => 'datetime',
        'featured_until' => 'datetime',
        'negotiable' => 'boolean',
        'verified_company' => 'boolean',
        'benefits' => 'array',
        'scam_flags' => 'array',
        'seo_data' => 'array',
    ];

    /**
     * Get the organization that owns the job.
     */
    public function organization()
    {
        return $this->belongsTo(Organization::class, 'organization_id');
    }

    /**
     * Get the applications for the job.
     */
    public function applications()
    {
        return $this->hasMany(Application::class, 'job_id');
    }

    /**
     * Get the interactions for the job.
     */
    public function interactions()
    {
        return $this->morphMany(Interaction::class, 'target');
    }

    /**
     * Get the saved items for the job.
     */
    public function savedItems()
    {
        return $this->morphMany(SavedItem::class, 'item');
    }

    /**
     * Check if job is active.
     */
    public function isActive(): bool
    {
        return $this->status === 'active';
    }

    /**
     * Check if job is featured.
     */
    public function isFeatured(): bool
    {
        return $this->featured_until && $this->featured_until->isFuture();
    }

    /**
     * Check if job has expired.
     */
    public function isExpired(): bool
    {
        return $this->application_deadline && $this->application_deadline->isPast();
    }

    /**
     * Get job location display.
     */
    public function getLocationDisplayAttribute(): string
    {
        if ($this->location_city && $this->location_region) {
            return $this->location_city . ', ' . $this->location_region;
        }
        return $this->location_region ?: $this->location_city ?: 'Remote';
    }

    /**
     * Get salary range display.
     */
    public function getSalaryDisplayAttribute(): ?string
    {
        if (!$this->salary_min && !$this->salary_max) {
            return null;
        }

        $min = $this->salary_min ? number_format($this->salary_min) : '0';
        $max = $this->salary_max ? number_format($this->salary_max) : '';

        if ($this->salary_min && $this->salary_max) {
            return $min . ' - ' . $max . ' ' . $this->currency;
        } elseif ($this->salary_min) {
            return 'From ' . $min . ' ' . $this->currency;
        } elseif ($this->salary_max) {
            return 'Up to ' . $max . ' ' . $this->currency;
        }

        return null;
    }

    /**
     * Scope for active jobs.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope for featured jobs.
     */
    public function scopeFeatured($query)
    {
        return $query->where('featured_until', '>', now());
    }

    /**
     * Scope for jobs by location.
     */
    public function scopeByLocation($query, $location)
    {
        return $query->where('location_city', 'like', '%' . $location . '%')
                    ->orWhere('location_region', 'like', '%' . $location . '%');
    }

    /**
     * Scope for jobs by employment type.
     */
    public function scopeByEmploymentType($query, $type)
    {
        return $query->where('employment_type', $type);
    }
}