<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Organization extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'organizations_optimized';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'slug',
        'description',
        'sector',
        'contact_email',
        'contact_phone',
        'address',
        'website',
        'logo',
        'banner_image',
        'registration_number',
        'tax_id',
        'owner_user_id',
        'status',
        'subscription_plan',
        'subscription_expires_at',
        'branding',
        'settings',
        'domains',
        'custom_features',
        'seo_data',
        'marketing_settings',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'subscription_expires_at' => 'datetime',
        'branding' => 'array',
        'settings' => 'array',
        'domains' => 'array',
        'custom_features' => 'array',
        'seo_data' => 'array',
        'marketing_settings' => 'array',
    ];

    /**
     * Get the owner of the organization.
     */
    public function owner()
    {
        return $this->belongsTo(User::class, 'owner_user_id');
    }

    /**
     * Get the jobs for the organization.
     */
    public function jobs()
    {
        return $this->hasMany(Job::class, 'organization_id');
    }

    /**
     * Get the applications for the organization's jobs.
     */
    public function applications()
    {
        return $this->hasManyThrough(Application::class, Job::class, 'organization_id', 'job_id');
    }

    /**
     * Get the interactions for the organization.
     */
    public function interactions()
    {
        return $this->morphMany(Interaction::class, 'target');
    }

    /**
     * Get the transactions for the organization.
     */
    public function transactions()
    {
        return $this->hasMany(Transaction::class, 'organization_id');
    }

    /**
     * Check if organization is active.
     */
    public function isActive(): bool
    {
        return $this->status === 'active';
    }

    /**
     * Check if organization has active subscription.
     */
    public function hasActiveSubscription(): bool
    {
        return $this->subscription_expires_at && $this->subscription_expires_at->isFuture();
    }

    /**
     * Get organization logo URL.
     */
    public function getLogoUrlAttribute(): ?string
    {
        return $this->logo ? asset('storage/' . $this->logo) : null;
    }

    /**
     * Get organization banner URL.
     */
    public function getBannerUrlAttribute(): ?string
    {
        return $this->banner_image ? asset('storage/' . $this->banner_image) : null;
    }

    /**
     * Get organization initials.
     */
    public function getInitialsAttribute(): string
    {
        $name = trim($this->name);
        if (empty($name)) {
            return 'O';
        }

        $words = preg_split('/\s+/', $name);
        if (count($words) >= 2) {
            return strtoupper(substr($words[0], 0, 1) . substr($words[1], 0, 1));
        }

        return strtoupper(substr($name, 0, 2));
    }
}