<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class SavedItem extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'saved_items_optimized';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'item_type',
        'item_id',
        'query_data',
        'name',
        'description',
        'is_auto_refresh',
        'last_checked_at',
        'notification_frequency',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'query_data' => 'array',
        'is_auto_refresh' => 'boolean',
        'last_checked_at' => 'datetime',
    ];

    /**
     * Get the user who saved the item.
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Get the saved item (polymorphic relationship).
     */
    public function item()
    {
        return $this->morphTo();
    }

    /**
     * Check if item is a job.
     */
    public function isJob(): bool
    {
        return $this->item_type === 'job';
    }

    /**
     * Check if item is a company/organization.
     */
    public function isCompany(): bool
    {
        return $this->item_type === 'company';
    }

    /**
     * Check if item is a search.
     */
    public function isSearch(): bool
    {
        return $this->item_type === 'search';
    }

    /**
     * Check if item is a filter.
     */
    public function isFilter(): bool
    {
        return $this->item_type === 'filter';
    }

    /**
     * Check if auto refresh is enabled.
     */
    public function hasAutoRefresh(): bool
    {
        return $this->is_auto_refresh;
    }

    /**
     * Get item type label.
     */
    public function getTypeLabelAttribute(): string
    {
        return match($this->item_type) {
            'job' => 'Job',
            'company' => 'Company',
            'search' => 'Search',
            'filter' => 'Filter',
            default => ucfirst($this->item_type),
        };
    }

    /**
     * Scope for saved items by type.
     */
    public function scopeByType($query, $type)
    {
        return $query->where('item_type', $type);
    }

    /**
     * Scope for saved items by user.
     */
    public function scopeByUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    /**
     * Scope for auto refresh items.
     */
    public function scopeAutoRefresh($query)
    {
        return $query->where('is_auto_refresh', true);
    }

    /**
     * Scope for items that need checking.
     */
    public function scopeNeedsChecking($query)
    {
        return $query->where('is_auto_refresh', true)
                    ->where(function ($q) {
                        $q->whereNull('last_checked_at')
                          ->orWhere('last_checked_at', '<', now()->subHours(24));
                    });
    }
}