<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Session extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'sessions_optimized';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'id',
        'user_id',
        'ip_address',
        'user_agent',
        'payload',
        'last_activity',
        'is_active',
        'device_type',
        'location',
        'remember_token',
        'login_method',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'is_active' => 'boolean',
        'last_activity' => 'integer',
    ];

    /**
     * Indicates if the model's ID is auto-incrementing.
     *
     * @var bool
     */
    public $incrementing = false;

    /**
     * The data type of the auto-incrementing ID.
     *
     * @var string
     */
    protected $keyType = 'string';

    /**
     * Get the user that owns the session.
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Check if session is active.
     */
    public function isActive(): bool
    {
        return $this->is_active;
    }

    /**
     * Check if session is expired.
     */
    public function isExpired(): bool
    {
        $lifetime = config('session.lifetime', 120); // minutes
        return $this->last_activity < (time() - ($lifetime * 60));
    }

    /**
     * Get device type label.
     */
    public function getDeviceTypeLabelAttribute(): string
    {
        return match($this->device_type) {
            'desktop' => 'Desktop',
            'mobile' => 'Mobile',
            'tablet' => 'Tablet',
            default => 'Unknown',
        };
    }

    /**
     * Get login method label.
     */
    public function getLoginMethodLabelAttribute(): string
    {
        return match($this->login_method) {
            'email' => 'Email',
            'social' => 'Social Login',
            'admin' => 'Admin Login',
            default => 'Standard',
        };
    }

    /**
     * Scope for active sessions.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope for sessions by user.
     */
    public function scopeByUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    /**
     * Scope for expired sessions.
     */
    public function scopeExpired($query)
    {
        $lifetime = config('session.lifetime', 120);
        return $query->where('last_activity', '<', time() - ($lifetime * 60));
    }
}