<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Spatie\Permission\Traits\HasRoles;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable, HasRoles;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'users_optimized';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'account_type',
        'role',
        'status',
        'preferred_language',
        'avatar',
        'bio',
        'location',
        'phone',
        'date_of_birth',
        'gender',
        'current_position',
        'years_experience',
        'industry',
        'highest_education_level',
        'education_details',
        'profile_data',
        'email_notifications',
        'privacy_settings',
        'preferences',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the organization associated with the user.
     */
    public function organization()
    {
        return $this->hasOne(Organization::class, 'owner_user_id');
    }

    /**
     * Get the company associated with the user (legacy compatibility).
     */
    public function company()
    {
        return $this->hasOne(Organization::class, 'owner_user_id');
    }

    /**
     * Get the profile associated with the user.
     * Returns a hasOne relationship to UserProfile model.
     */
    public function profile()
    {
        return $this->hasOne(UserProfile::class);
    }
    
    /**
     * Get the profile data (either from UserProfile or from user fields directly).
     * This provides compatibility between old profile system and new direct user fields.
     */
    public function getProfileDataAttribute()
    {
        // If there's a UserProfile, merge it with user fields
        if ($this->profile) {
            return array_merge($this->profile->toArray(), $this->toArray());
        }
        
        // Otherwise, return user fields as profile data
        return $this->toArray();
    }
    
    /**
     * Accessor for avatar that checks user avatar field directly.
     * Since profile data is stored directly in users table, no relationship needed.
     */
    public function getAvatarUrlAttribute()
    {
        // Check user avatar field directly
        if ($this->avatar) {
            return asset('storage/' . $this->avatar);
        }
        
        // Return null if no avatar found
        return null;
    }

    /**
     * Get the applications for the user.
     */
    public function applications()
    {
        return $this->hasMany(Application::class);
    }

    /**
     * Check if user is admin (legacy method for backward compatibility).
     */
    public function getIsAdminAttribute()
    {
        // Check simplified role system first
        if ($this->role === 'admin' || $this->account_type === 'admin') {
            return true;
        }

        // Fallback to email check for backward compatibility
        return $this->email === 'admin@hirecameroon.com' || $this->id === 1;
    }

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'account_type' => 'string',
    ];

    public function organizationsOwned()
    {
        return $this->hasMany(Organization::class, 'owner_user_id');
    }

    public function companiesOwned()
    {
        return $this->hasMany(Organization::class, 'owner_user_id');
    }

    /**
     * Check if user has applied to a specific job
     */
    public function hasAppliedToJob(Job $job): bool
    {
        return $this->applications()->where('job_id', $job->id)->exists();
    }

    /**
     * Check if user is a company account
     */
    public function isCompany(): bool
    {
        return $this->account_type === 'company';
    }

    /**
     * Check if user is a personal account
     */
    public function isPersonal(): bool
    {
        return $this->account_type === 'personal';
    }

    /**
     * Check if user can post jobs
     */
    public function canPostJobs(): bool
    {
        return $this->isCompany() || $this->hasRole('admin');
    }

    /**
     * Get user initials from name
     */
    public function getInitialsAttribute(): string
    {
        $name = trim($this->name);
        if (empty($name)) {
            return 'U';
        }
        
        $words = preg_split('/\s+/', $name);
        if (count($words) >= 2) {
            return strtoupper(substr($words[0], 0, 1) . substr($words[1], 0, 1));
        }
        
        return strtoupper(substr($name, 0, 2));
    }

    /**
     * Get a color for the avatar based on user ID (consistent colors)
     */
    public function getAvatarColorAttribute(): string
    {
        $colors = [
            '#FF6B6B', '#4ECDC4', '#45B7D1', '#96CEB4', '#FFEAA7',
            '#DDA0DD', '#98D8C8', '#F7DC6F', '#BB8FCE', '#85C1E9',
            '#F8C471', '#82E0AA', '#F1948A', '#85C1E9', '#F7DC6F'
        ];
        
        $index = $this->id % count($colors);
        return $colors[$index];
    }
}
