<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Str;
use App\Models\Application;

class ApplicationSubmitted extends Notification implements ShouldQueue
{
    use Queueable;

    public Application $application;

    /**
     * Create a new notification instance.
     *
     * @param Application $application
     * @return void
     */
    public function __construct(Application $application)
    {
        $this->application = $application;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        return ['mail'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {
        $application = $this->application;
        $job = $application->job;
        $company = $job->company;

        return (new MailMessage)
            ->subject("New Application Received: {$job->title}")
            ->greeting("Hello {$company->name} Team!")
            ->line("You have received a new application for the position: **{$job->title}**")
            ->line("**Applicant:** {$application->user->name}")
            ->line("**Applied on:** {$application->created_at->format('F j, Y \a\t g:i A')}")
            ->when($application->cover_letter, function ($mail) use ($application) {
                return $mail->line("**Cover Letter:** " . Str::limit($application->cover_letter, 100));
            })
            ->action('View Application', route('dashboard.applications'))
            ->line('Please review the application and update the status accordingly.')
            ->salutation('Best regards, Hire Cameroon Team');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable)
    {
        return [
            'application_id' => $this->application->id,
            'job_title' => $this->application->job->title,
            'applicant_name' => $this->application->user->name,
            'company_name' => $this->application->job->company->name,
        ];
    }
}
