<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use App\Models\Subscription;

class SubscriptionExpiring extends Notification implements ShouldQueue
{
    use Queueable;

    public Subscription $subscription;
    public int $daysRemaining;

    /**
     * Create a new notification instance.
     *
     * @param Subscription $subscription
     * @param int $daysRemaining
     * @return void
     */
    public function __construct(Subscription $subscription, int $daysRemaining)
    {
        $this->subscription = $subscription;
        $this->daysRemaining = $daysRemaining;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        return ['mail'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {
        $subscription = $this->subscription;
        $company = $subscription->company;

        return (new MailMessage)
            ->subject("Your Hire Cameroon Subscription Expires Soon")
            ->greeting("Hello {$company->name} Team!")
            ->line("Your {$subscription->plan_name} subscription will expire in {$this->daysRemaining} day" . ($this->daysRemaining === 1 ? '' : 's') . ".")
            ->line("**Expiration Date:** {$subscription->ends_at->format('F j, Y')}")
            ->line("**Current Plan:** {$subscription->plan_name}")
            ->line("Don't lose access to premium features! Renew your subscription to continue:")
            ->line("• Posting unlimited jobs")
            ->line("• Advanced ATS features")
            ->line("• Priority support")
            ->action('Renew Subscription', route('subscriptions.subscribe', $company))
            ->line('After expiration, your account will revert to free tier limitations.')
            ->salutation('Best regards, Hire Cameroon Team');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable)
    {
        return [
            'subscription_id' => $this->subscription->id,
            'plan_name' => $this->subscription->plan_name,
            'ends_at' => $this->subscription->ends_at,
            'days_remaining' => $this->daysRemaining,
        ];
    }
}
