<?php

namespace App\Policies;

use App\Models\Company;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class CompanyPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any companies.
     */
    public function viewAny(User $user)
    {
        return true; // Anyone can view companies
    }

    /**
     * Determine whether the user can view the company.
     */
    public function view(User $user, Company $company)
    {
        return true; // Anyone can view companies
    }

    /**
     * Determine whether the user can create companies.
     */
    public function create(User $user)
    {
        return $user->hasRole(['admin', 'employer']);
    }

    /**
     * Determine whether the user can update the company.
     */
    public function update(User $user, Company $company)
    {
        // Admin can update any company
        if ($user->hasRole('admin')) {
            return true;
        }

        // Company owner can update their own company
        return $user->hasRole('employer') &&
               $company->owner_user_id === $user->id;
    }

    /**
     * Determine whether the user can delete the company.
     */
    public function delete(User $user, Company $company)
    {
        // Admin can delete any company
        if ($user->hasRole('admin')) {
            return true;
        }

        // Company owner can delete their own company
        return $user->hasRole('employer') &&
               $company->owner_user_id === $user->id;
    }

    /**
     * Determine whether the user can manage jobs for the company.
     */
    public function manageJobs(User $user, Company $company)
    {
        // Admin can manage any company's jobs
        if ($user->hasRole('admin')) {
            return true;
        }

        // Company owner can manage their own company's jobs
        return $user->hasRole('employer') &&
               $company->owner_user_id === $user->id;
    }

    /**
     * Determine whether the user can manage subscriptions for the company.
     */
    public function manageSubscriptions(User $user, Company $company)
    {
        // Admin can manage any company's subscriptions
        if ($user->hasRole('admin')) {
            return true;
        }

        // Company owner can manage their own company's subscriptions
        return $user->hasRole('employer') &&
               $company->owner_user_id === $user->id;
    }

    /**
     * Determine whether the user can view applications for the company.
     */
    public function viewApplications(User $user, Company $company)
    {
        // Admin can view any company's applications
        if ($user->hasRole('admin')) {
            return true;
        }

        // Company owner can view their own company's applications
        return $user->hasRole('employer') &&
               $company->owner_user_id === $user->id;
    }
}