<?php

namespace App\Services;

use App\Models\AuditLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class AuditService
{
    public static function log($action, $resourceType, $resourceId = null, $description = null, $oldValues = null, $newValues = null, $metadata = [])
    {
        $request = request();

        AuditLog::create([
            'user_id' => Auth::id(),
            'action' => $action,
            'resource_type' => $resourceType,
            'resource_id' => $resourceId,
            'description' => $description,
            'old_values' => $oldValues,
            'new_values' => $newValues,
            'metadata' => array_merge($metadata, [
                'url' => $request ? $request->fullUrl() : null,
                'method' => $request ? $request->method() : null,
            ]),
            'ip_address' => $request ? $request->ip() : null,
            'user_agent' => $request ? $request->userAgent() : null,
            'session_id' => session()->getId(),
        ]);
    }

    public static function logLogin()
    {
        self::log('login', 'user', Auth::id(), 'User logged into the system');
    }

    public static function logLogout()
    {
        self::log('logout', 'user', Auth::id(), 'User logged out of the system');
    }

    public static function logCreate($resourceType, $resourceId, $description = null, $newValues = null)
    {
        self::log('create', $resourceType, $resourceId, $description ?: "Created new {$resourceType}", null, $newValues);
    }

    public static function logUpdate($resourceType, $resourceId, $description = null, $oldValues = null, $newValues = null)
    {
        self::log('update', $resourceType, $resourceId, $description ?: "Updated {$resourceType}", $oldValues, $newValues);
    }

    public static function logDelete($resourceType, $resourceId, $description = null, $oldValues = null)
    {
        self::log('delete', $resourceType, $resourceId, $description ?: "Deleted {$resourceType}", $oldValues, null);
    }

    public static function logView($resourceType, $resourceId, $description = null)
    {
        self::log('view', $resourceType, $resourceId, $description ?: "Viewed {$resourceType}");
    }

    public static function logExport($resourceType, $description = null, $metadata = [])
    {
        self::log('export', $resourceType, null, $description ?: "Exported {$resourceType} data", null, null, $metadata);
    }

    public static function logImport($resourceType, $description = null, $metadata = [])
    {
        self::log('import', $resourceType, null, $description ?: "Imported {$resourceType} data", null, null, $metadata);
    }

    public static function logAdminAction($action, $description, $metadata = [])
    {
        self::log($action, 'admin', null, $description, null, null, $metadata);
    }

    public static function getAuditLogs($filters = [], $perPage = 20)
    {
        $query = AuditLog::with('user')
            ->orderBy('created_at', 'desc');

        if (!empty($filters['days'])) {
            $query->recent($filters['days']);
        }

        if (!empty($filters['action'])) {
            $query->byAction($filters['action']);
        }

        if (!empty($filters['resource_type'])) {
            $query->byResource($filters['resource_type'], $filters['resource_id'] ?? null);
        }

        if (!empty($filters['user_id'])) {
            $query->byUser($filters['user_id']);
        }

        if (!empty($filters['search'])) {
            $search = $filters['search'];
            $query->where(function ($q) use ($search) {
                $q->where('description', 'like', "%{$search}%")
                  ->orWhere('action', 'like', "%{$search}%")
                  ->orWhere('resource_type', 'like', "%{$search}%")
                  ->orWhereHas('user', function ($userQuery) use ($search) {
                      $userQuery->where('name', 'like', "%{$search}%")
                               ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        return $query->paginate($perPage);
    }

    public static function getAuditStats($days = 30)
    {
        $startDate = now()->subDays($days);

        return [
            'total_logs' => AuditLog::where('created_at', '>=', $startDate)->count(),
            'unique_users' => AuditLog::where('created_at', '>=', $startDate)->distinct('user_id')->count(),
            'actions_breakdown' => AuditLog::selectRaw('action, COUNT(*) as count')
                ->where('created_at', '>=', $startDate)
                ->groupBy('action')
                ->orderBy('count', 'desc')
                ->get(),
            'resources_breakdown' => AuditLog::selectRaw('resource_type, COUNT(*) as count')
                ->where('created_at', '>=', $startDate)
                ->whereNotNull('resource_type')
                ->groupBy('resource_type')
                ->orderBy('count', 'desc')
                ->get(),
            'daily_activity' => AuditLog::selectRaw('DATE(created_at) as date, COUNT(*) as count')
                ->where('created_at', '>=', $startDate)
                ->groupBy('date')
                ->orderBy('date')
                ->get()
                ->pluck('count', 'date'),
        ];
    }

    public static function getUserActivity($userId, $days = 30)
    {
        return AuditLog::byUser($userId)
            ->recent($days)
            ->orderBy('created_at', 'desc')
            ->get();
    }

    public static function getResourceHistory($resourceType, $resourceId, $limit = 50)
    {
        return AuditLog::byResource($resourceType, $resourceId)
            ->with('user')
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }

    public static function exportAuditLogs($filters = [])
    {
        $logs = self::getAuditLogs($filters, 10000); // Large limit for export

        $csvData = [];
        $csvData[] = ['ID', 'User', 'Email', 'Action', 'Resource Type', 'Resource ID', 'Description', 'IP Address', 'Timestamp'];

        foreach ($logs as $log) {
            $csvData[] = [
                $log->id,
                $log->user ? $log->user->name : 'System',
                $log->user ? $log->user->email : 'N/A',
                $log->action,
                $log->resource_type,
                $log->resource_id,
                $log->description,
                $log->ip_address,
                $log->created_at->format('Y-m-d H:i:s'),
            ];
        }

        return $csvData;
    }
}