<?php

namespace App\Services;

use App\Models\User;
use App\Models\Company;
use App\Services\Email\EmailService;
use App\Services\Sms\SmsService;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class CommunicationService
{
    protected $emailService;
    protected $smsService;
    protected array $channelSettings;

    public function __construct(EmailService $emailService, SmsService $smsService)
    {
        $this->emailService = $emailService;
        $this->smsService = $smsService;
        $this->channelSettings = config('communication.settings', []);
    }

    /**
     * Send a message using the appropriate channel
     */
    public function sendMessage(string $channel, string $template, array $data, ?User $user = null, ?string $recipient = null, ?string $recipientPhone = null): bool
    {
        try {
            $this->logAttempt($channel, $template, $recipient, $recipientPhone);

            switch ($channel) {
                case 'email':
                    return $this->sendEmail($template, $data, $user, $recipient);
                
                case 'sms':
                    return $this->sendSms($template, $data, $user, $recipientPhone);
                
                default:
                    Log::error("Unsupported communication channel: {$channel}");
                    return false;
            }
        } catch (\Exception $e) {
            Log::error("Communication send failed: " . $e->getMessage(), [
                'channel' => $channel,
                'template' => $template,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send email with multiple provider support
     */
    public function sendEmail(string $template, array $data, ?User $user = null, ?string $recipient = null): bool
    {
        $emailProvider = $this->getEmailProvider();
        
        if (!$recipient && $user) {
            $recipient = $user->email;
        }

        if (!$recipient) {
            Log::error('No email recipient provided');
            return false;
        }

        // Check throttling
        if ($this->isThrottled($recipient, 'email')) {
            Log::warning("Email throttled for: {$recipient}");
            return false;
        }

        $result = match($emailProvider) {
            'phpmailer' => $this->emailService->sendWithPhpMailer($template, $data, $recipient),
            'smtp' => $this->emailService->sendWithSmtp($template, $data, $recipient),
            'mailgun' => $this->emailService->sendWithMailgun($template, $data, $recipient),
            'ses' => $this->emailService->sendWithSes($template, $data, $recipient),
            'postmark' => $this->emailService->sendWithPostmark($template, $data, $recipient),
            default => $this->emailService->sendWithSmtp($template, $data, $recipient),
        };

        if ($result) {
            $this->recordSuccess($recipient, 'email');
            Log::info("Email sent successfully via {$emailProvider} to {$recipient}");
        } else {
            $this->recordFailure($recipient, 'email');
            Log::error("Email failed to send via {$emailProvider} to {$recipient}");
        }

        return $result;
    }

    /**
     * Send SMS with multiple provider support
     */
    public function sendSms(string $template, array $data, ?User $user = null, ?string $recipientPhone = null): bool
    {
        $smsProvider = $this->getSmsProvider();
        
        if (!$recipientPhone && $user && $user->phone) {
            $recipientPhone = $user->phone;
        }

        if (!$recipientPhone) {
            Log::error('No phone number provided for SMS');
            return false;
        }

        // Check throttling
        if ($this->isThrottled($recipientPhone, 'sms')) {
            Log::warning("SMS throttled for: {$recipientPhone}");
            return false;
        }

        $result = match($smsProvider) {
            'twilio' => $this->smsService->sendWithTwilio($template, $data, $recipientPhone),
            'messagebird' => $this->smsService->sendWithMessageBird($template, $data, $recipientPhone),
            'nexmo' => $this->smsService->sendWithNexmo($template, $data, $recipientPhone),
            'africatalking' => $this->smsService->sendWithAfricaTalking($template, $data, $recipientPhone),
            default => false,
        };

        if ($result) {
            $this->recordSuccess($recipientPhone, 'sms');
            Log::info("SMS sent successfully via {$smsProvider} to {$recipientPhone}");
        } else {
            $this->recordFailure($recipientPhone, 'sms');
            Log::error("SMS failed to send via {$smsProvider} to {$recipientPhone}");
        }

        return $result;
    }

    /**
     * Send application status notification
     */
    public function sendApplicationStatusNotification($application, string $status): void
    {
        $data = [
            'application' => $application,
            'job' => $application->job,
            'company' => $application->job->company,
            'user' => $application->user,
            'status' => $status,
        ];

        // Send email
        $this->sendMessage('email', 'application_status', $data, $application->user);
        
        // Send SMS if user has phone and wants SMS notifications
        if ($application->user->phone && $this->userWantsSmsNotifications($application->user)) {
            $this->sendMessage('sms', 'application_status', $data, $application->user);
        }
    }

    /**
     * Send password reset notification
     */
    public function sendPasswordResetNotification(User $user, string $resetCode): void
    {
        $data = [
            'user' => $user,
            'reset_code' => $resetCode,
        ];

        // Send email with reset link
        $this->sendMessage('email', 'password_reset', $data, $user);
        
        // Send SMS with code if enabled
        if ($user->phone && $this->userWantsSmsNotifications($user)) {
            $this->sendMessage('sms', 'password_reset', $data, $user);
        }
    }

    /**
     * Send job alerts to subscribed users
     */
    public function sendJobAlert($job, array $subscribedUsers): void
    {
        foreach ($subscribedUsers as $user) {
            $data = [
                'job' => $job,
                'user' => $user,
            ];

            // Send email alert
            $this->sendMessage('email', 'job_alert', $data, $user);
            
            // Send SMS if enabled and user has phone
            if ($user->phone && $this->userWantsSmsNotifications($user)) {
                $this->sendMessage('sms', 'job_alert', $data, $user);
            }
        }
    }

    /**
     * Get the current email provider
     */
    protected function getEmailProvider(): string
    {
        return config('communication.providers.email.default', 'smtp');
    }

    /**
     * Get the current SMS provider
     */
    protected function getSmsProvider(): string
    {
        return config('communication.providers.sms.default', 'twilio');
    }

    /**
     * Check if user wants SMS notifications
     */
    protected function userWantsSmsNotifications(User $user): bool
    {
        return $user->notification_preferences['sms'] ?? false;
    }

    /**
     * Check throttling limits
     */
    protected function isThrottled(string $recipient, string $channel): bool
    {
        $limits = config('communication.throttling', []);
        $limit = match($channel) {
            'email' => $limits['emails_per_hour'] ?? 100,
            'sms' => $limits['sms_per_hour'] ?? 20,
            default => 10,
        };

        $cacheKey = "communication:throttle:{$channel}:{$recipient}";
        $count = Cache::get($cacheKey, 0);
        
        if ($count >= $limit) {
            return true;
        }

        Cache::put($cacheKey, $count + 1, now()->addHour());
        return false;
    }

    /**
     * Record successful communication
     */
    protected function recordSuccess(string $recipient, string $channel): void
    {
        if (!config('communication.logging.enabled', true)) {
            return;
        }

        DB::table('communication_logs')->insert([
            'recipient' => $recipient,
            'channel' => $channel,
            'status' => 'sent',
            'sent_at' => now(),
        ]);
    }

    /**
     * Record failed communication
     */
    protected function recordFailure(string $recipient, string $channel): void
    {
        if (!config('communication.logging.enabled', true)) {
            return;
        }

        DB::table('communication_logs')->insert([
            'recipient' => $recipient,
            'channel' => $channel,
            'status' => 'failed',
            'sent_at' => now(),
        ]);
    }

    /**
     * Log communication attempt
     */
    protected function logAttempt(string $channel, string $template, ?string $recipient, ?string $recipientPhone): void
    {
        Log::info("Communication attempt", [
            'channel' => $channel,
            'template' => $template,
            'recipient_email' => $recipient,
            'recipient_phone' => $recipientPhone,
        ]);
    }

    /**
     * Get communication statistics
     */
    public function getStats(?string $period = '7 days'): array
    {
        $since = now()->subDays(7);
        
        return [
            'emails_sent' => DB::table('communication_logs')
                ->where('channel', 'email')
                ->where('sent_at', '>=', $since)
                ->count(),
            'sms_sent' => DB::table('communication_logs')
                ->where('channel', 'sms')
                ->where('sent_at', '>=', $since)
                ->count(),
            'failed_emails' => DB::table('communication_logs')
                ->where('channel', 'email')
                ->where('status', 'failed')
                ->where('sent_at', '>=', $since)
                ->count(),
            'failed_sms' => DB::table('communication_logs')
                ->where('channel', 'sms')
                ->where('status', 'failed')
                ->where('sent_at', '>=', $since)
                ->count(),
        ];
    }
}