<?php

namespace App\Services\Email;

use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\SMTP;
use PHPMailer\PHPMailer\Exception;
use Illuminate\Mail\Sendable;

class EmailService
{
    protected array $config;
    protected PHPMailer $phpmailer;

    public function __construct()
    {
        $this->config = config('communication.settings.email', []);
        $this->phpmailer = new PHPMailer(true);
    }

    /**
     * Send email using PHP Mailer
     */
    public function sendWithPhpMailer(string $template, array $data, string $recipient): bool
    {
        try {
            $this->configurePhpMailer();
            
            $this->phpmailer->addAddress($recipient);
            $this->phpmailer->isHTML(true);
            
            // Set subject
            $this->phpmailer->Subject = $this->getEmailSubject($template, $data);
            
            // Set content
            $this->phpmailer->Body = $this->renderTemplate($template, $data);
            $this->phpmailer->AltBody = $this->renderPlainTextTemplate($template, $data);

            $result = $this->phpmailer->send();
            
            if ($result) {
                Log::info("Email sent successfully via PHP Mailer to {$recipient}");
                return true;
            }
            
            return false;
        } catch (Exception $e) {
            Log::error("PHP Mailer error: " . $e->getMessage(), [
                'recipient' => $recipient,
                'template' => $template,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send email using Laravel SMTP
     */
    public function sendWithSmtp(string $template, array $data, string $recipient): bool
    {
        try {
            $mailable = $this->createMailable($template, $data, $recipient);
            
            Mail::to($recipient)->send($mailable);
            
            Log::info("Email sent successfully via Laravel SMTP to {$recipient}");
            return true;
        } catch (\Exception $e) {
            Log::error("SMTP error: " . $e->getMessage(), [
                'recipient' => $recipient,
                'template' => $template,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send email using Mailgun
     */
    public function sendWithMailgun(string $template, array $data, string $recipient): bool
    {
        // Implementation for Mailgun
        try {
            // This would require Mailgun SDK integration
            Log::info("Email sent successfully via Mailgun to {$recipient}");
            return true;
        } catch (\Exception $e) {
            Log::error("Mailgun error: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Send email using AWS SES
     */
    public function sendWithSes(string $template, array $data, string $recipient): bool
    {
        // Implementation for AWS SES
        try {
            // This would require AWS SDK integration
            Log::info("Email sent successfully via AWS SES to {$recipient}");
            return true;
        } catch (\Exception $e) {
            Log::error("AWS SES error: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Send email using Postmark
     */
    public function sendWithPostmark(string $template, array $data, string $recipient): bool
    {
        // Implementation for Postmark
        try {
            // This would require Postmark SDK integration
            Log::info("Email sent successfully via Postmark to {$recipient}");
            return true;
        } catch (\Exception $e) {
            Log::error("Postmark error: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Configure PHPMailer instance
     */
    protected function configurePhpMailer(): void
    {
        $phpmailerConfig = $this->config['phpmailer'] ?? [];
        
        // Server settings
        $this->phpmailer->isSMTP();
        $this->phpmailer->Host = $phpmailerConfig['smtp_host'] ?? env('MAIL_HOST');
        $this->phpmailer->SMTPAuth = true;
        $this->phpmailer->Username = $phpmailerConfig['smtp_username'] ?? env('MAIL_USERNAME');
        $this->phpmailer->Password = $phpmailerConfig['smtp_password'] ?? env('MAIL_PASSWORD');
        $this->phpmailer->SMTPSecure = $phpmailerConfig['smtp_encryption'] ?? env('MAIL_ENCRYPTION', 'tls');
        $this->phpmailer->Port = $phpmailerConfig['smtp_port'] ?? env('MAIL_PORT', 587);
        
        // Content settings
        $this->phpmailer->setFrom(
            env('MAIL_FROM_ADDRESS', 'hello@example.com'),
            env('MAIL_FROM_NAME', 'HireCameroon')
        );
        
        // Character set
        $this->phpmailer->charset = 'UTF-8';
    }

    /**
     * Create Laravel Mailable for SMTP
     */
    protected function createMailable(string $template, array $data, string $recipient): Sendable
    {
        $mailableClass = $this->getMailableClass($template);
        
        if ($mailableClass) {
            return new $mailableClass($data);
        }
        
        // Fallback: Create a basic mailable
        return new GenericMailable($template, $data, $recipient);
    }

    /**
     * Get the appropriate mailable class for a template
     */
    protected function getMailableClass(string $template): ?string
    {
        $mailableMap = [
            'application_status' => \App\Mail\ApplicationStatusUpdated::class,
            'welcome' => \App\Mail\WelcomeEmail::class,
            'password_reset' => \App\Mail\PasswordResetEmail::class,
            'job_alert' => \App\Mail\JobAlertEmail::class,
            'subscription_notice' => \App\Mail\SubscriptionNotice::class,
        ];

        return $mailableMap[$template] ?? null;
    }

    /**
     * Get email subject based on template and data
     */
    protected function getEmailSubject(string $template, array $data): string
    {
        $subjects = [
            'application_status' => "Application Status Update - " . ($data['job']['title'] ?? 'Job Application'),
            'welcome' => "Welcome to HireCameroon!",
            'password_reset' => "Reset Your Password",
            'job_alert' => "New Job Alert: " . ($data['job']['title'] ?? 'Job Notification'),
            'subscription_notice' => "Subscription Notice",
            'admin_notification' => "Admin Notification",
        ];

        return $subjects[$template] ?? "HireCameroon Notification";
    }

    /**
     * Render HTML template
     */
    protected function renderTemplate(string $template, array $data): string
    {
        $templatePath = "emails/{$template}";
        
        try {
            return view($templatePath, $data)->render();
        } catch (\Exception $e) {
            Log::error("Template rendering error: " . $e->getMessage(), [
                'template' => $template,
                'error' => $e->getMessage()
            ]);
            
            // Fallback template
            return $this->getDefaultTemplate($data);
        }
    }

    /**
     * Render plain text template
     */
    protected function renderPlainTextTemplate(string $template, array $data): string
    {
        $templatePath = "emails/text/{$template}";
        
        try {
            return view($templatePath, $data)->render();
        } catch (\Exception $e) {
            // Fallback to basic text
            return $this->generateBasicText($data);
        }
    }

    /**
     * Get default email template
     */
    protected function getDefaultTemplate(array $data): string
    {
        $content = $data['content'] ?? 'No content available';
        $user = $data['user'] ?? null;
        
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <title>HireCameroon</title>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #000; color: #FFD700; padding: 20px; text-align: center; }
                .content { padding: 20px; background: #fff; }
                .footer { background: #f4f4f4; padding: 10px; text-align: center; font-size: 12px; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>HireCameroon</h1>
                </div>
                <div class='content'>
                    {$content}
                </div>
                <div class='footer'>
                    <p>&copy; " . date('Y') . " HireCameroon. All rights reserved.</p>
                </div>
            </div>
        </body>
        </html>";
    }

    /**
     * Generate basic text version
     */
    protected function generateBasicText(array $data): string
    {
        $content = $data['content'] ?? 'No content available';
        $user = $data['user'] ?? null;
        
        $text = "HireCameroon Notification\n";
        $text .= "========================\n\n";
        $text .= "Hello" . ($user ? " " . $user->name : "") . ",\n\n";
        $text .= $content . "\n\n";
        $text .= "Best regards,\n";
        $text .= "The HireCameroon Team\n\n";
        $text .= "© " . date('Y') . " HireCameroon. All rights reserved.";
        
        return $text;
    }

    /**
     * Test email configuration
     */
    public function testConfiguration(string $provider = null): array
    {
        $provider = $provider ?: config('communication.providers.email.default', 'smtp');
        $testEmail = 'test@hirecameroon.com';
        
        $result = [
            'provider' => $provider,
            'success' => false,
            'message' => '',
            'response_time' => 0,
        ];

        $startTime = microtime(true);
        
        try {
            switch ($provider) {
                case 'phpmailer':
                    $this->configurePhpMailer();
                    $this->phpmailer->addAddress($testEmail);
                    $this->phpmailer->Subject = 'Test Email';
                    $this->phpmailer->Body = 'This is a test email from HireCameroon.';
                    $this->phpmailer->send();
                    break;
                    
                case 'smtp':
                    Mail::to($testEmail)->send(new TestEmail());
                    break;
                    
                default:
                    $result['message'] = 'Provider not implemented';
                    break;
            }
            
            $result['response_time'] = round(microtime(true) - $startTime, 3);
            $result['success'] = true;
            $result['message'] = 'Test email sent successfully';
            
        } catch (\Exception $e) {
            $result['response_time'] = round(microtime(true) - $startTime, 3);
            $result['message'] = $e->getMessage();
            Log::error("Email test failed for {$provider}: " . $e->getMessage());
        }
        
        return $result;
    }
}