<?php

namespace App\Services;

use App\Models\Job;
use App\Models\Company;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class JobVerificationService
{
    /**
     * Analyze a job posting for potential scam indicators
     */
    public function analyzeJob(Job $job): array
    {
        $flags = [];
        $riskScore = 0;

        // Check for unrealistic salary
        if ($this->hasUnrealisticSalary($job)) {
            $flags[] = 'unrealistic_salary';
            $riskScore += 30;
        }

        // Check for missing company website
        if ($this->hasNoCompanyWebsite($job)) {
            $flags[] = 'no_company_website';
            $riskScore += 20;
        }

        // Check for generic description
        if ($this->hasGenericDescription($job)) {
            $flags[] = 'generic_description';
            $riskScore += 15;
        }

        // Check for urgent hiring language
        if ($this->hasUrgentHiring($job)) {
            $flags[] = 'urgent_hiring';
            $riskScore += 25;
        }

        // Check for work from home only
        if ($this->isWorkFromHomeOnly($job)) {
            $flags[] = 'work_from_home_only';
            $riskScore += 10;
        }

        // Check for no benefits mentioned
        if ($this->hasNoBenefits($job)) {
            $flags[] = 'no_benefits';
            $riskScore += 10;
        }

        // Check for no requirements
        if ($this->hasNoRequirements($job)) {
            $flags[] = 'no_requirements';
            $riskScore += 5;
        }

        // Check for suspicious email patterns
        if ($this->hasSuspiciousEmail($job)) {
            $flags[] = 'suspicious_email';
            $riskScore += 15;
        }

        // Check company reputation
        $companyRisk = $this->analyzeCompanyReputation($job->company);
        $riskScore += $companyRisk['score'];
        $flags = array_merge($flags, $companyRisk['flags']);

        return [
            'flags' => $flags,
            'risk_score' => min($riskScore, 100),
            'is_suspicious' => $riskScore >= 50,
            'recommendation' => $this->getRecommendation($riskScore),
        ];
    }

    /**
     * Check if salary is unrealistically high
     */
    private function hasUnrealisticSalary(Job $job): bool
    {
        if (!$job->salary_max) return false;

        $sectorAverages = [
            'technology' => 300000,
            'finance' => 250000,
            'healthcare' => 200000,
            'education' => 150000,
            'retail' => 100000,
            'logistics' => 120000,
            'manufacturing' => 130000,
            'construction' => 140000,
            'agriculture' => 90000,
            'energy' => 180000,
        ];

        $maxSalary = $sectorAverages[$job->sector_type] ?? 150000;
        return $job->salary_max > $maxSalary * 2; // More than double the sector average
    }

    /**
     * Check if company has no website
     */
    private function hasNoCompanyWebsite(Job $job): bool
    {
        return empty($job->company->website);
    }

    /**
     * Check for generic job description
     */
    private function hasGenericDescription(Job $job): bool
    {
        $genericPhrases = [
            'we are looking for',
            'responsible for',
            'must have',
            'excellent communication skills',
            'team player',
            'fast paced environment',
        ];

        $description = strtolower($job->description);
        $genericCount = 0;

        foreach ($genericPhrases as $phrase) {
            if (str_contains($description, $phrase)) {
                $genericCount++;
            }
        }

        return $genericCount >= 3; // Too many generic phrases
    }

    /**
     * Check for urgent hiring language
     */
    private function hasUrgentHiring(Job $job): bool
    {
        $urgentPhrases = [
            'urgent',
            'immediate start',
            'start immediately',
            'hiring now',
            'apply today',
            'limited time',
            'act fast',
        ];

        $description = strtolower($job->description . ' ' . $job->requirements);
        foreach ($urgentPhrases as $phrase) {
            if (str_contains($description, $phrase)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if job is work from home only
     */
    private function isWorkFromHomeOnly(Job $job): bool
    {
        $description = strtolower($job->description . ' ' . $job->requirements);
        return str_contains($description, 'work from home') ||
               str_contains($description, 'remote work') ||
               str_contains($description, 'home based');
    }

    /**
     * Check if no benefits are mentioned
     */
    private function hasNoBenefits(Job $job): bool
    {
        return empty($job->benefits);
    }

    /**
     * Check if no requirements are specified
     */
    private function hasNoRequirements(Job $job): bool
    {
        return empty($job->requirements);
    }

    /**
     * Check for suspicious email patterns
     */
    private function hasSuspiciousEmail(Job $job): bool
    {
        // This would check the company's contact email for suspicious patterns
        // For now, return false as we don't have email validation logic
        return false;
    }

    /**
     * Analyze company reputation
     */
    private function analyzeCompanyReputation(Company $company): array
    {
        $flags = [];
        $score = 0;

        // Check if company is newly created (less than 30 days old)
        if ($company->created_at->diffInDays(now()) < 30) {
            $flags[] = 'new_company';
            $score += 15;
        }

        // Check if company has no description
        if (empty($company->description)) {
            $flags[] = 'no_company_description';
            $score += 10;
        }

        // Check if company has no logo
        if (empty($company->logo)) {
            $flags[] = 'no_company_logo';
            $score += 5;
        }

        return [
            'flags' => $flags,
            'score' => $score,
        ];
    }

    /**
     * Get recommendation based on risk score
     */
    private function getRecommendation(int $riskScore): string
    {
        if ($riskScore >= 80) {
            return 'Reject immediately - high risk of scam';
        } elseif ($riskScore >= 60) {
            return 'Manual review required - moderate to high risk';
        } elseif ($riskScore >= 40) {
            return 'Flag for monitoring - low to moderate risk';
        } else {
            return 'Approve - low risk';
        }
    }

    /**
     * Auto-moderate a job based on analysis
     */
    public function autoModerate(Job $job): void
    {
        $analysis = $this->analyzeJob($job);

        if ($analysis['is_suspicious']) {
            $job->flagAsScam($analysis['flags']);
        }

        // Auto-approve low risk jobs
        if ($analysis['risk_score'] < 30) {
            $job->markAsVerified();
        }

        Log::info('Job auto-moderated', [
            'job_id' => $job->id,
            'risk_score' => $analysis['risk_score'],
            'flags' => $analysis['flags'],
            'recommendation' => $analysis['recommendation'],
        ]);
    }

    /**
     * Verify company website exists
     */
    public function verifyCompanyWebsite(string $url): bool
    {
        try {
            $response = Http::timeout(10)->get($url);
            return $response->successful();
        } catch (\Exception $e) {
            Log::warning('Failed to verify company website', [
                'url' => $url,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }
}