<?php

namespace App\Services\Sms;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use GuzzleHttp\Client;

class SmsService
{
    protected array $config;
    protected array $templates;

    public function __construct()
    {
        $this->config = config('communication.settings.sms', []);
        $this->templates = config('communication.sms_templates', []);
    }

    /**
     * Send SMS using Twilio
     */
    public function sendWithTwilio(string $template, array $data, string $phoneNumber): bool
    {
        try {
            $twilioConfig = $this->config['twilio'] ?? [];
            $message = $this->compileTemplate($template, $data);
            
            $url = "https://api.twilio.com/2010-04-01/Accounts/{$twilioConfig['sid']}/Messages.json";
            
            $response = Http::withBasicAuth(
                $twilioConfig['sid'],
                $twilioConfig['token']
            )->post($url, [
                'From' => $twilioConfig['from'],
                'To' => $this->formatPhoneNumber($phoneNumber),
                'Body' => $message,
            ]);

            if ($response->successful()) {
                Log::info("SMS sent successfully via Twilio to {$phoneNumber}");
                return true;
            }

            Log::error("Twilio SMS failed", [
                'phone' => $phoneNumber,
                'status' => $response->status(),
                'response' => $response->body()
            ]);
            
            return false;
        } catch (\Exception $e) {
            Log::error("Twilio SMS error: " . $e->getMessage(), [
                'phone' => $phoneNumber,
                'template' => $template,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send SMS using MessageBird
     */
    public function sendWithMessageBird(string $template, array $data, string $phoneNumber): bool
    {
        try {
            $messagebirdConfig = $this->config['messagebird'] ?? [];
            $message = $this->compileTemplate($template, $data);
            
            $url = "https://rest.messagebird.com/messages";
            
            $response = Http::withHeaders([
                'Authorization' => 'AccessKey: ' . $messagebirdConfig['api_key'],
                'Content-Type' => 'application/json',
            ])->post($url, [
                'originator' => $messagebirdConfig['originator'],
                'recipients' => [$this->formatPhoneNumber($phoneNumber)],
                'body' => $message,
            ]);

            if ($response->successful()) {
                Log::info("SMS sent successfully via MessageBird to {$phoneNumber}");
                return true;
            }

            Log::error("MessageBird SMS failed", [
                'phone' => $phoneNumber,
                'status' => $response->status(),
                'response' => $response->body()
            ]);
            
            return false;
        } catch (\Exception $e) {
            Log::error("MessageBird SMS error: " . $e->getMessage(), [
                'phone' => $phoneNumber,
                'template' => $template,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send SMS using Nexmo
     */
    public function sendWithNexmo(string $template, array $data, string $phoneNumber): bool
    {
        try {
            $nexmoConfig = $this->config['nexmo'] ?? [];
            $message = $this->compileTemplate($template, $data);
            
            $client = new Client();
            $response = $client->post('https://rest.nexmo.com/sms/json', [
                'form_params' => [
                    'api_key' => $nexmoConfig['api_key'],
                    'api_secret' => $nexmoConfig['api_secret'],
                    'from' => $nexmoConfig['from'],
                    'to' => $this->formatPhoneNumber($phoneNumber),
                    'text' => $message,
                ],
            ]);

            $body = json_decode($response->getBody(), true);
            
            if (isset($body['messages'][0]['status']) && $body['messages'][0]['status'] === '0') {
                Log::info("SMS sent successfully via Nexmo to {$phoneNumber}");
                return true;
            }

            Log::error("Nexmo SMS failed", [
                'phone' => $phoneNumber,
                'response' => $body
            ]);
            
            return false;
        } catch (\Exception $e) {
            Log::error("Nexmo SMS error: " . $e->getMessage(), [
                'phone' => $phoneNumber,
                'template' => $template,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send SMS using Africa Talking
     */
    public function sendWithAfricaTalking(string $template, array $data, string $phoneNumber): bool
    {
        try {
            $africaConfig = $this->config['africatalking'] ?? [];
            $message = $this->compileTemplate($template, $data);
            
            $client = new Client();
            $response = $client->post('https://api.africastalking.com/version1/messaging', [
                'form_params' => [
                    'username' => $africaConfig['username'],
                    'to' => $this->formatPhoneNumber($phoneNumber),
                    'message' => $message,
                    'from' => $africaConfig['from'],
                ],
                'headers' => [
                    'apiKey' => $africaConfig['api_key'],
                    'Content-Type' => 'application/x-www-form-urlencoded',
                ],
            ]);

            $body = json_decode($response->getBody(), true);
            
            if (isset($body['SMSMessageData']['Recipients'][0]['status']) && 
                $body['SMSMessageData']['Recipients'][0]['status'] === 'Success') {
                Log::info("SMS sent successfully via Africa Talking to {$phoneNumber}");
                return true;
            }

            Log::error("Africa Talking SMS failed", [
                'phone' => $phoneNumber,
                'response' => $body
            ]);
            
            return false;
        } catch (\Exception $e) {
            Log::error("Africa Talking SMS error: " . $e->getMessage(), [
                'phone' => $phoneNumber,
                'template' => $template,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Compile SMS template with data
     */
    protected function compileTemplate(string $template, array $data): string
    {
        $templateString = $this->templates[$template] ?? 'Message: No template found';
        
        // Replace placeholders with actual data
        foreach ($data as $key => $value) {
            if (is_scalar($value)) {
                $templateString = str_replace("{{$key}}", $value, $templateString);
            }
        }
        
        return $templateString;
    }

    /**
     * Format phone number for international format
     */
    protected function formatPhoneNumber(string $phoneNumber): string
    {
        // Remove all non-numeric characters
        $phone = preg_replace('/[^0-9]/', '', $phoneNumber);
        
        // Add +237 (Cameroon country code) if not present
        if (substr($phone, 0, 1) !== '+' && substr($phone, 0, 3) !== '237') {
            if (substr($phone, 0, 1) === '0') {
                $phone = '237' . substr($phone, 1);
            } else {
                $phone = '237' . $phone;
            }
        }
        
        return '+' . $phone;
    }

    /**
     * Test SMS configuration
     */
    public function testConfiguration(string $provider = null): array
    {
        $provider = $provider ?: config('communication.providers.sms.default', 'twilio');
        $testPhone = '+237612345678';
        
        $result = [
            'provider' => $provider,
            'success' => false,
            'message' => '',
            'response_time' => 0,
        ];

        $startTime = microtime(true);
        
        try {
            $success = false;
            
            switch ($provider) {
                case 'twilio':
                    $success = $this->sendWithTwilio('test', ['message' => 'Test SMS'], $testPhone);
                    break;
                case 'messagebird':
                    $success = $this->sendWithMessageBird('test', ['message' => 'Test SMS'], $testPhone);
                    break;
                case 'nexmo':
                    $success = $this->sendWithNexmo('test', ['message' => 'Test SMS'], $testPhone);
                    break;
                case 'africatalking':
                    $success = $this->sendWithAfricaTalking('test', ['message' => 'Test SMS'], $testPhone);
                    break;
                default:
                    $result['message'] = 'Provider not implemented';
                    break;
            }
            
            $result['response_time'] = round(microtime(true) - $startTime, 3);
            $result['success'] = $success;
            $result['message'] = $success ? 'Test SMS sent successfully' : 'Test SMS failed';
            
        } catch (\Exception $e) {
            $result['response_time'] = round(microtime(true) - $startTime, 3);
            $result['message'] = $e->getMessage();
            Log::error("SMS test failed for {$provider}: " . $e->getMessage());
        }
        
        return $result;
    }

    /**
     * Get SMS delivery status
     */
    public function getDeliveryStatus(string $messageId, string $provider): ?array
    {
        try {
            switch ($provider) {
                case 'twilio':
                    return $this->getTwilioStatus($messageId);
                case 'messagebird':
                    return $this->getMessageBirdStatus($messageId);
                case 'nexmo':
                    return $this->getNexmoStatus($messageId);
                case 'africatalking':
                    return $this->getAfricaTalkingStatus($messageId);
                default:
                    return null;
            }
        } catch (\Exception $e) {
            Log::error("Status check failed: " . $e->getMessage());
            return null;
        }
    }

    protected function getTwilioStatus(string $messageId): ?array
    {
        $config = $this->config['twilio'] ?? [];
        $url = "https://api.twilio.com/2010-04-01/Accounts/{$config['sid']}/Messages/{$messageId}.json";
        
        $response = Http::withBasicAuth($config['sid'], $config['token'])->get($url);
        
        if ($response->successful()) {
            return $response->json();
        }
        
        return null;
    }

    protected function getMessageBirdStatus(string $messageId): ?array
    {
        $config = $this->config['messagebird'] ?? [];
        $url = "https://rest.messagebird.com/messages/{$messageId}";
        
        $response = Http::withHeaders([
            'Authorization' => 'AccessKey: ' . $config['api_key'],
        ])->get($url);
        
        if ($response->successful()) {
            return $response->json();
        }
        
        return null;
    }

    protected function getNexmoStatus(string $messageId): ?array
    {
        // Nexmo status checking would require additional implementation
        return null;
    }

    protected function getAfricaTalkingStatus(string $messageId): ?array
    {
        // Africa Talking status checking would require additional implementation
        return null;
    }
}