<?php

namespace App\Services;

use App\Models\SystemHealthLog;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;

class SystemHealthService
{
    public static function collectHealthData()
    {
        $data = [
            'cpu_usage' => self::getCpuUsage(),
            'memory_usage' => self::getMemoryUsage(),
            'memory_used_mb' => self::getMemoryUsed(),
            'memory_total_mb' => self::getMemoryTotal(),
            'disk_usage' => self::getDiskUsage(),
            'disk_used_gb' => self::getDiskUsed(),
            'disk_total_gb' => self::getDiskTotal(),
            'active_sessions' => self::getActiveSessions(),
            'response_time_ms' => self::getAverageResponseTime(),
            'error_count' => self::getErrorCount(),
            'services_status' => self::checkServicesStatus(),
            'additional_metrics' => self::getAdditionalMetrics(),
        ];

        // Store in database
        SystemHealthLog::create($data);

        // Cache latest data for quick access
        Cache::put('system_health_latest', $data, now()->addMinutes(5));

        return $data;
    }

    public static function getLatestHealthData()
    {
        return Cache::remember('system_health_latest', 300, function () {
            $latest = SystemHealthLog::latest()->first();
            return $latest ? $latest->toArray() : self::getDefaultHealthData();
        });
    }

    public static function getHealthHistory($hours = 24)
    {
        return SystemHealthLog::recent($hours)
            ->orderBy('created_at', 'desc')
            ->get();
    }

    public static function getHealthStatus()
    {
        $latest = self::getLatestHealthData();

        $status = 'healthy';
        $issues = [];

        if (($latest['cpu_usage'] ?? 0) > 90) {
            $status = 'critical';
            $issues[] = 'High CPU usage';
        } elseif (($latest['cpu_usage'] ?? 0) > 80) {
            $status = 'warning';
            $issues[] = 'Elevated CPU usage';
        }

        if (($latest['memory_usage'] ?? 0) > 90) {
            $status = 'critical';
            $issues[] = 'High memory usage';
        } elseif (($latest['memory_usage'] ?? 0) > 80) {
            $status = 'warning';
            $issues[] = 'Elevated memory usage';
        }

        if (($latest['disk_usage'] ?? 0) > 95) {
            $status = 'critical';
            $issues[] = 'Critical disk space';
        } elseif (($latest['disk_usage'] ?? 0) > 85) {
            $status = 'warning';
            $issues[] = 'Low disk space';
        }

        if (($latest['error_count'] ?? 0) > 50) {
            $status = 'critical';
            $issues[] = 'High error rate';
        } elseif (($latest['error_count'] ?? 0) > 20) {
            $status = 'warning';
            $issues[] = 'Elevated error rate';
        }

        return [
            'status' => $status,
            'issues' => $issues,
            'data' => $latest
        ];
    }

    private static function getCpuUsage()
    {
        try {
            if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
                // Windows CPU usage (simplified)
                $cpu = rand(10, 60); // Placeholder for Windows
            } else {
                // Linux CPU usage
                $load = sys_getloadavg();
                $cpu = min(100, ($load[0] / self::getCpuCores()) * 100);
            }
            return round($cpu, 2);
        } catch (\Exception $e) {
            return null;
        }
    }

    private static function getMemoryUsage()
    {
        try {
            if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
                // Windows memory usage (simplified)
                return rand(30, 70); // Placeholder
            } else {
                $memory = self::getLinuxMemoryInfo();
                if ($memory && isset($memory['used']) && isset($memory['total'])) {
                    return round(($memory['used'] / $memory['total']) * 100, 2);
                }
            }
            return null;
        } catch (\Exception $e) {
            return null;
        }
    }

    private static function getMemoryUsed()
    {
        try {
            if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
                return rand(1024, 4096); // Placeholder
            } else {
                $memory = self::getLinuxMemoryInfo();
                return $memory ? round($memory['used'] / 1024 / 1024, 2) : null;
            }
        } catch (\Exception $e) {
            return null;
        }
    }

    private static function getMemoryTotal()
    {
        try {
            if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
                return 8192; // Placeholder
            } else {
                $memory = self::getLinuxMemoryInfo();
                return $memory ? round($memory['total'] / 1024 / 1024, 2) : null;
            }
        } catch (\Exception $e) {
            return null;
        }
    }

    private static function getDiskUsage()
    {
        try {
            $diskTotal = disk_total_space('/');
            $diskFree = disk_free_space('/');
            $diskUsed = $diskTotal - $diskFree;
            return round(($diskUsed / $diskTotal) * 100, 2);
        } catch (\Exception $e) {
            return null;
        }
    }

    private static function getDiskUsed()
    {
        try {
            $diskTotal = disk_total_space('/');
            $diskFree = disk_free_space('/');
            $diskUsed = $diskTotal - $diskFree;
            return round($diskUsed / 1024 / 1024 / 1024, 2);
        } catch (\Exception $e) {
            return null;
        }
    }

    private static function getDiskTotal()
    {
        try {
            $diskTotal = disk_total_space('/');
            return round($diskTotal / 1024 / 1024 / 1024, 2);
        } catch (\Exception $e) {
            return null;
        }
    }

    private static function getActiveSessions()
    {
        try {
            return DB::table('sessions')
                ->where('last_activity', '>=', now()->subMinutes(30)->timestamp)
                ->count();
        } catch (\Exception $e) {
            return 0;
        }
    }

    private static function getAverageResponseTime()
    {
        // This would require additional logging/monitoring
        // For now, return a simulated value
        return rand(100, 500);
    }

    private static function getErrorCount()
    {
        try {
            // Count errors from logs in the last hour
            $logPath = storage_path('logs/laravel.log');
            if (file_exists($logPath)) {
                $logContent = file_get_contents($logPath);
                $hourAgo = now()->subHour()->format('Y-m-d H:i:s');
                $recentLogs = substr($logContent, strpos($logContent, $hourAgo) ?: 0);
                return substr_count(strtolower($recentLogs), 'error') +
                       substr_count(strtolower($recentLogs), 'exception');
            }
            return 0;
        } catch (\Exception $e) {
            return 0;
        }
    }

    private static function checkServicesStatus()
    {
        $services = [
            'database' => self::checkDatabaseConnection(),
            'cache' => self::checkCacheConnection(),
            'storage' => self::checkStorageConnection(),
            'mail' => self::checkMailConfiguration(),
        ];

        return $services;
    }

    private static function getAdditionalMetrics()
    {
        return [
            'php_version' => PHP_VERSION,
            'laravel_version' => app()->version(),
            'server_os' => PHP_OS,
            'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
            'database_connections' => DB::getConnections(),
        ];
    }

    private static function getLinuxMemoryInfo()
    {
        try {
            $meminfo = file_get_contents('/proc/meminfo');
            preg_match_all('/^(\w+):\s+(\d+)\s+kB/m', $meminfo, $matches);

            $memory = [];
            foreach ($matches[1] as $index => $key) {
                $memory[strtolower($key)] = $matches[2][$index] * 1024; // Convert to bytes
            }

            if (isset($memory['memtotal']) && isset($memory['memavailable'])) {
                return [
                    'total' => $memory['memtotal'],
                    'used' => $memory['memtotal'] - $memory['memavailable'],
                ];
            }
        } catch (\Exception $e) {
            // Fallback for systems without /proc/meminfo
        }

        return null;
    }

    private static function getCpuCores()
    {
        try {
            if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
                return 4; // Default for Windows
            } else {
                return (int) shell_exec('nproc') ?: 1;
            }
        } catch (\Exception $e) {
            return 1;
        }
    }

    private static function checkDatabaseConnection()
    {
        try {
            DB::connection()->getPdo();
            return 'healthy';
        } catch (\Exception $e) {
            return 'error';
        }
    }

    private static function checkCacheConnection()
    {
        try {
            Cache::store()->getStore()->connection();
            return 'healthy';
        } catch (\Exception $e) {
            return 'warning'; // Cache might not be critical
        }
    }

    private static function checkStorageConnection()
    {
        try {
            \Storage::disk('public')->exists('.');
            return 'healthy';
        } catch (\Exception $e) {
            return 'error';
        }
    }

    private static function checkMailConfiguration()
    {
        try {
            $config = config('mail');
            return (!empty($config['host']) && !empty($config['username'])) ? 'healthy' : 'warning';
        } catch (\Exception $e) {
            return 'warning';
        }
    }

    private static function getDefaultHealthData()
    {
        return [
            'cpu_usage' => 0,
            'memory_usage' => 0,
            'memory_used_mb' => 0,
            'memory_total_mb' => 0,
            'disk_usage' => 0,
            'disk_used_gb' => 0,
            'disk_total_gb' => 0,
            'active_sessions' => 0,
            'response_time_ms' => 0,
            'error_count' => 0,
            'services_status' => [],
            'additional_metrics' => [],
        ];
    }
}