<?php

namespace App\Services;

use App\Models\UserActivityLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class UserActivityService
{
    public static function log($action, $resourceType = null, $resourceId = null, $description = null, $metadata = [])
    {
        if (!Auth::check()) {
            return;
        }

        $request = request();

        UserActivityLog::create([
            'user_id' => Auth::id(),
            'action' => $action,
            'resource_type' => $resourceType,
            'resource_id' => $resourceId,
            'description' => $description,
            'metadata' => array_merge($metadata, [
                'url' => $request ? $request->fullUrl() : null,
                'method' => $request ? $request->method() : null,
            ]),
            'ip_address' => $request ? $request->ip() : null,
            'user_agent' => $request ? $request->userAgent() : null,
        ]);
    }

    public static function logLogin()
    {
        self::log('login', null, null, 'User logged in');
    }

    public static function logLogout()
    {
        self::log('logout', null, null, 'User logged out');
    }

    public static function logJobView($jobId)
    {
        self::log('view_job', 'job', $jobId, 'Viewed job details');
    }

    public static function logJobApplication($jobId, $applicationId)
    {
        self::log('apply_job', 'job', $jobId, 'Applied for job', [
            'application_id' => $applicationId
        ]);
    }

    public static function logCompanyView($companyId)
    {
        self::log('view_company', 'company', $companyId, 'Viewed company profile');
    }

    public static function logProfileUpdate()
    {
        self::log('update_profile', 'user', Auth::id(), 'Updated user profile');
    }

    public static function logJobPost($jobId)
    {
        self::log('post_job', 'job', $jobId, 'Posted new job');
    }

    public static function getUserActivity($userId, $limit = 50)
    {
        return UserActivityLog::where('user_id', $userId)
            ->with('user')
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }

    public static function getRecentActivity($days = 7, $limit = 100)
    {
        return UserActivityLog::with('user')
            ->recent($days)
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }

    public static function getActivityStats($days = 30)
    {
        $startDate = now()->subDays($days);

        return [
            'total_activities' => UserActivityLog::where('created_at', '>=', $startDate)->count(),
            'unique_users' => UserActivityLog::where('created_at', '>=', $startDate)->distinct('user_id')->count(),
            'top_actions' => UserActivityLog::selectRaw('action, COUNT(*) as count')
                ->where('created_at', '>=', $startDate)
                ->groupBy('action')
                ->orderBy('count', 'desc')
                ->limit(10)
                ->get(),
            'daily_activity' => UserActivityLog::selectRaw('DATE(created_at) as date, COUNT(*) as count')
                ->where('created_at', '>=', $startDate)
                ->groupBy('date')
                ->orderBy('date')
                ->get()
                ->pluck('count', 'date'),
        ];
    }
}