<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Config;

class RolesAndPlansSeeder extends Seeder
{
    public function run(): void
    {
        $now = now();

        // Roles
        foreach (['admin', 'employer', 'recruiter', 'job_seeker'] as $role) {
            DB::table('roles')->updateOrInsert(['name' => $role], [
                'guard_name' => 'web',
                'created_at' => $now,
                'updated_at' => $now,
            ]);
        }

        // Permissions (minimal placeholders)
        foreach ([
            'manage_companies', 'manage_jobs', 'view_applicants', 'manage_applicants', 'manage_subscriptions'
        ] as $perm) {
            DB::table('permissions')->updateOrInsert(['name' => $perm], [
                'guard_name' => 'web',
                'created_at' => $now,
                'updated_at' => $now,
            ]);
        }

        // Create default admin user
        $roleIds = DB::table('roles')->pluck('id', 'name');
        $adminUser = \App\Models\User::firstOrCreate(
            ['email' => 'admin@hirecameroon.com'],
            [
                'name' => 'System Administrator',
                'password' => bcrypt('admin123'),
                'email_verified_at' => $now,
                'created_at' => $now,
                'updated_at' => $now,
            ]
        );

        // Assign admin role to the admin user
        if ($adminUser && isset($roleIds['admin'])) {
            $adminUser->roles()->sync([$roleIds['admin']]);
        }

        // Map permissions to roles (admin gets all; employer gets company/job/subscription; recruiter gets applicant)
        $roleIds = DB::table('roles')->pluck('id', 'name');
        $permIds = DB::table('permissions')->pluck('id', 'name');

        $attach = function (string $role, array $perms) use ($roleIds, $permIds) {
            if (!isset($roleIds[$role])) return;
            foreach ($perms as $p) {
                if (isset($permIds[$p])) {
                    DB::table('permission_role')->updateOrInsert([
                        'permission_id' => $permIds[$p],
                        'role_id' => $roleIds[$role],
                    ], []);
                }
            }
        };

        $allPerms = array_keys($permIds->toArray());
        $attach('admin', $allPerms);
        $attach('employer', ['manage_companies', 'manage_jobs', 'manage_subscriptions', 'view_applicants']);
        $attach('recruiter', ['view_applicants', 'manage_applicants']);
        $attach('job_seeker', []);

        // Default plans (config-based)
        // Using subscriptions table when a company subscribes; here we just document available plans in config.
        $plans = [
            [
                'name' => 'Basic',
                'features' => ['5 job posts', 'ATS basic', 'Email alerts'],
                'price_xaf' => 0,
            ],
            [
                'name' => 'Pro',
                'features' => ['50 job posts', 'ATS advanced', 'Featured listings (3)'],
                'price_xaf' => 15000,
            ],
        ];

        Config::set('hirecameroon.plans', $plans);
    }
}


