<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>HireCameroon - Installation Wizard</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }

        .container {
            background: white;
            border-radius: 12px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            max-width: 600px;
            width: 100%;
            overflow: hidden;
        }

        .header {
            background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }

        .header h1 {
            font-size: 2.5rem;
            margin-bottom: 10px;
            font-weight: 700;
        }

        .header p {
            opacity: 0.9;
            font-size: 1.1rem;
        }

        .content {
            padding: 40px;
        }

        .form-section, .step {
            margin-bottom: 30px;
            padding: 20px;
            border-radius: 8px;
            border-left: 4px solid #e0e0e0;
        }

        .form-section { border-left-color: #3498db; background: #f8f9fa; }

        .step.checking { border-left-color: #f39c12; background: #fff3cd; }
        .step.success { border-left-color: #27ae60; background: #d4edda; }
        .step.error { border-left-color: #e74c3c; background: #f8d7da; }

        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: 600;
            color: #333;
        }

        .form-group input, .form-group select {
            width: 100%;
            padding: 10px;
            border: 2px solid #ddd;
            border-radius: 4px;
            font-size: 1rem;
            transition: border-color 0.3s;
        }

        .form-group input:focus, .form-group select:focus {
            outline: none;
            border-color: #3498db;
        }

        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
        }

        .checkbox-group {
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .checkbox-group input[type="checkbox"] {
            width: auto;
        }

        .step h3 {
            font-size: 1.2rem;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
        }

        .step .icon {
            margin-right: 10px;
            font-size: 1.5rem;
        }

        .step .details {
            color: #666;
            font-size: 0.9rem;
            margin-top: 10px;
        }

        .progress-bar {
            width: 100%;
            height: 8px;
            background: #e0e0e0;
            border-radius: 4px;
            overflow: hidden;
            margin: 20px 0;
        }

        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, #27ae60, #2ecc71);
            width: 0%;
            transition: width 0.3s ease;
        }

        .btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            padding: 12px 30px;
            border-radius: 6px;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
            transition: transform 0.2s;
        }

        .btn:hover {
            transform: translateY(-2px);
        }

        .btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }

        .btn-secondary {
            background: #6c757d;
        }

        .requirements {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 20px;
            margin: 20px 0;
        }

        .requirements h4 {
            margin-bottom: 15px;
            color: #333;
        }

        .requirement {
            display: flex;
            justify-content: space-between;
            padding: 8px 0;
            border-bottom: 1px solid #eee;
        }

        .requirement:last-child {
            border-bottom: none;
        }

        .status {
            font-weight: 600;
        }

        .status.pass { color: #27ae60; }
        .status.fail { color: #e74c3c; }
        .status.checking { color: #f39c12; }

        .error-details {
            background: #f8d7da;
            border: 1px solid #f5c6cb;
            color: #721c24;
            padding: 15px;
            border-radius: 6px;
            margin-top: 15px;
        }

        .success-message {
            text-align: center;
            padding: 40px 20px;
        }

        .success-message h2 {
            color: #27ae60;
            margin-bottom: 20px;
        }

        .next-steps {
            background: #e8f5e8;
            border-radius: 8px;
            padding: 20px;
            margin: 20px 0;
        }

        .next-steps h4 {
            color: #27ae60;
            margin-bottom: 15px;
        }

        .next-steps ul {
            padding-left: 20px;
        }

        .next-steps li {
            margin-bottom: 8px;
            color: #333;
        }

        .tab-buttons {
            display: flex;
            margin-bottom: 30px;
            background: #f8f9fa;
            border-radius: 8px;
            padding: 5px;
        }

        .tab-btn {
            flex: 1;
            padding: 10px 20px;
            border: none;
            background: transparent;
            border-radius: 6px;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s;
        }

        .tab-btn.active {
            background: white;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            color: #3498db;
        }

        .tab-content {
            display: none;
        }

        .tab-content.active {
            display: block;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🚀 HireCameroon</h1>
            <p>Installation Wizard</p>
        </div>

        <div class="content">
            <div class="tab-buttons">
                <button class="tab-btn active" onclick="switchTab('config')">Configuration</button>
                <button class="tab-btn" onclick="switchTab('install')">Installation</button>
            </div>

            <div id="config-tab" class="tab-content active">
                <form id="config-form" onsubmit="saveConfiguration(event)">
                    <div class="form-section">
                        <h3>🗄️ Database Configuration</h3>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="db_host">Database Host</label>
                                <input type="text" id="db_host" name="db_host" value="localhost" required>
                            </div>
                            <div class="form-group">
                                <label for="db_port">Database Port</label>
                                <input type="number" id="db_port" name="db_port" value="3306" required>
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="db_name">Database Name</label>
                                <input type="text" id="db_name" name="db_name" value="bestgqmc_hirecameroon" required>
                            </div>
                            <div class="form-group">
                                <label for="db_user">Database Username</label>
                                <input type="text" id="db_user" name="db_user" value="bestgqmc_hirecameroon" required>
                            </div>
                        </div>
                        <div class="form-group">
                            <label for="db_pass">Database Password</label>
                            <input type="password" id="db_pass" name="db_pass" placeholder="Enter database password" required>
                        </div>
                    </div>

                    <div class="form-section">
                        <h3>📧 Email Configuration</h3>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="mail_driver">Mail Driver</label>
                                <select id="mail_driver" name="mail_driver">
                                    <option value="smtp">SMTP</option>
                                    <option value="mail">Mail</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label for="mail_host">SMTP Host</label>
                                <input type="text" id="mail_host" name="mail_host" placeholder="smtp.gmail.com">
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="mail_port">SMTP Port</label>
                                <input type="number" id="mail_port" name="mail_port" value="587">
                            </div>
                            <div class="form-group">
                                <label for="mail_encryption">Encryption</label>
                                <select id="mail_encryption" name="mail_encryption">
                                    <option value="tls">TLS</option>
                                    <option value="ssl">SSL</option>
                                    <option value="none">None</option>
                                </select>
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="mail_username">Email Username</label>
                                <input type="email" id="mail_username" name="mail_username" placeholder="your-email@gmail.com">
                            </div>
                            <div class="form-group">
                                <label for="mail_password">Email Password</label>
                                <input type="password" id="mail_password" name="mail_password" placeholder="Email password">
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="mail_from_address">From Email Address</label>
                                <input type="email" id="mail_from_address" name="mail_from_address" value="noreply@hirecameroon.com">
                            </div>
                            <div class="form-group">
                                <label for="mail_from_name">From Name</label>
                                <input type="text" id="mail_from_name" name="mail_from_name" value="HireCameroon">
                            </div>
                        </div>
                    </div>

                    <div class="form-section">
                        <h3>💳 Payment Configuration (Mesomb)</h3>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="mesomb_app_key">Application Key</label>
                                <input type="text" id="mesomb_app_key" name="mesomb_app_key" placeholder="Your Mesomb App Key">
                            </div>
                            <div class="form-group">
                                <label for="mesomb_access_key">Access Key</label>
                                <input type="text" id="mesomb_access_key" name="mesomb_access_key" placeholder="Your Mesomb Access Key">
                            </div>
                        </div>
                        <div class="form-group">
                            <label for="mesomb_secret_key">Secret Key</label>
                            <input type="text" id="mesomb_secret_key" name="mesomb_secret_key" placeholder="Your Mesomb Secret Key">
                        </div>
                    </div>

                    <div class="form-section">
                        <h3>👤 Admin Configuration</h3>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="admin_email">Admin Email</label>
                                <input type="email" id="admin_email" name="admin_email" value="admin@hirecameroon.com" required>
                            </div>
                            <div class="form-group">
                                <label for="admin_password">Admin Password</label>
                                <input type="password" id="admin_password" name="admin_password" value="admin123" required>
                            </div>
                        </div>
                        <div class="checkbox-group">
                            <input type="checkbox" id="create_admin" name="create_admin" checked>
                            <label for="create_admin">Create admin user with these credentials</label>
                        </div>
                    </div>

                    <div class="form-section">
                        <h3>⚙️ Application Settings</h3>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="app_name">Application Name</label>
                                <input type="text" id="app_name" name="app_name" value="HireCameroon" required>
                            </div>
                            <div class="form-group">
                                <label for="app_url">Application URL</label>
                                <input type="url" id="app_url" name="app_url" value="https://hirecameroon.com" required>
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="timezone">Timezone</label>
                                <select id="timezone" name="timezone">
                                    <option value="Africa/Douala">Africa/Douala (Cameroon)</option>
                                    <option value="UTC">UTC</option>
                                    <option value="Africa/Lagos">Africa/Lagos</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label for="currency">Default Currency</label>
                                <select id="currency" name="currency">
                                    <option value="XAF">XAF (CFA Franc)</option>
                                    <option value="USD">USD</option>
                                    <option value="EUR">EUR</option>
                                </select>
                            </div>
                        </div>
                    </div>

                    <div style="text-align: center; margin-top: 30px;">
                        <button type="submit" class="btn">Save Configuration & Test Connection</button>
                    </div>
                </form>
            </div>

            <div id="install-tab" class="tab-content">
                <div id="requirements" class="requirements">
                    <h4>📋 System Requirements Check</h4>
                    <div class="requirement">
                        <span>PHP Version (8.2+)</span>
                        <span class="status checking" id="php-check">Checking...</span>
                    </div>
                    <div class="requirement">
                        <span>Composer Dependencies</span>
                        <span class="status checking" id="composer-check">Checking...</span>
                    </div>
                    <div class="requirement">
                        <span>Storage Permissions</span>
                        <span class="status checking" id="storage-check">Checking...</span>
                    </div>
                    <div class="requirement">
                        <span>Database Connection</span>
                        <span class="status checking" id="db-check">Checking...</span>
                    </div>
                    <div class="requirement">
                        <span>Environment File</span>
                        <span class="status checking" id="env-check">Checking...</span>
                    </div>
                </div>

                <div class="progress-bar">
                    <div class="progress-fill" id="progress-fill"></div>
                </div>

                <div id="steps-container">
                    <!-- Steps will be dynamically added here -->
                </div>

                <div id="controls" style="text-align: center; margin-top: 30px;">
                    <button class="btn" id="install-btn" onclick="startInstallation()" disabled>Start Installation</button>
                </div>
            </div>
        </div>
    </div>

    <script>
        let currentStep = 0;
        let configData = {};
        const steps = [
            { name: 'Loading Laravel Framework', action: 'loadLaravel' },
            { name: 'Checking Database Connection', action: 'checkDatabase' },
            { name: 'Running Database Migrations', action: 'runMigrations' },
            { name: 'Seeding Database', action: 'seedDatabase' },
            { name: 'Optimizing Application', action: 'optimizeApp' },
            { name: 'Finalizing Installation', action: 'finalize' }
        ];

        function switchTab(tabName) {
            // Hide all tabs
            document.querySelectorAll('.tab-content').forEach(tab => {
                tab.classList.remove('active');
            });
            document.querySelectorAll('.tab-btn').forEach(btn => {
                btn.classList.remove('active');
            });

            // Show selected tab
            document.getElementById(tabName + '-tab').classList.add('active');
            event.target.classList.add('active');

            if (tabName === 'install') {
                checkRequirements();
            }
        }

        async function saveConfiguration(event) {
            event.preventDefault();

            const formData = new FormData(event.target);
            configData = Object.fromEntries(formData.entries());

            // Save configuration
            try {
                const response = await fetch('install/save-config.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(configData)
                });

                const result = await response.json();

                if (result.success) {
                    alert('Configuration saved successfully! You can now proceed to installation.');
                    document.getElementById('install-btn').disabled = false;
                    switchTab('install');
                } else {
                    alert('Failed to save configuration: ' + result.message);
                }
            } catch (error) {
                alert('Error saving configuration: ' + error.message);
            }
        }

        async function checkRequirements() {
            try {
                // Check PHP version
                const phpResponse = await fetch('install/check-requirements.php?check=php');
                const phpData = await phpResponse.json();
                updateRequirement('php-check', phpData.pass ? 'pass' : 'fail', phpData.version || 'Unknown');

                // Check Composer
                const composerResponse = await fetch('install/check-requirements.php?check=composer');
                const composerData = await composerResponse.json();
                updateRequirement('composer-check', composerData.pass ? 'pass' : 'fail', composerData.status);

                // Check Storage
                const storageResponse = await fetch('install/check-requirements.php?check=storage');
                const storageData = await storageResponse.json();
                updateRequirement('storage-check', storageData.pass ? 'pass' : 'fail', storageData.status);

                // Check Database
                const dbResponse = await fetch('install/check-requirements.php?check=database');
                const dbData = await dbResponse.json();
                updateRequirement('db-check', dbData.pass ? 'pass' : 'fail', dbData.status);

                // Check Environment
                const envResponse = await fetch('install/check-requirements.php?check=env');
                const envData = await envResponse.json();
                updateRequirement('env-check', envData.pass ? 'pass' : 'fail', envData.status);

            } catch (error) {
                console.error('Requirements check failed:', error);
            }
        }

        function updateRequirement(elementId, status, details) {
            const element = document.getElementById(elementId);
            element.className = `status ${status}`;
            element.textContent = details;
        }

        async function startInstallation() {
            document.getElementById('install-btn').disabled = true;
            document.getElementById('install-btn').textContent = 'Installing...';

            for (let i = 0; i < steps.length; i++) {
                currentStep = i;
                await runStep(steps[i]);
                updateProgress((i + 1) / steps.length * 100);
            }

            showSuccess();
        }

        async function runStep(step) {
            const stepElement = createStepElement(step.name, 'checking');
            document.getElementById('steps-container').appendChild(stepElement);

            try {
                const response = await fetch(`install/install-step.php?action=${step.action}`);
                const data = await response.json();

                if (data.success) {
                    updateStepStatus(stepElement, 'success', data.message);
                } else {
                    updateStepStatus(stepElement, 'error', data.message, data.details);
                    throw new Error(data.message);
                }
            } catch (error) {
                updateStepStatus(stepElement, 'error', 'Failed', error.message);
                throw error;
            }
        }

        function createStepElement(name, status) {
            const div = document.createElement('div');
            div.className = `step ${status}`;
            div.innerHTML = `
                <h3>
                    <span class="icon">${getStatusIcon(status)}</span>
                    ${name}
                </h3>
                <div class="details" id="step-details-${currentStep}">${getStatusText(status)}</div>
            `;
            return div;
        }

        function updateStepStatus(element, status, message, details = '') {
            element.className = `step ${status}`;
            const iconElement = element.querySelector('.icon');
            const detailsElement = element.querySelector('.details');

            iconElement.textContent = getStatusIcon(status);
            detailsElement.innerHTML = message;

            if (details) {
                detailsElement.innerHTML += `<div class="error-details">${details}</div>`;
            }
        }

        function getStatusIcon(status) {
            switch (status) {
                case 'checking': return '⏳';
                case 'success': return '✅';
                case 'error': return '❌';
                default: return '⏳';
            }
        }

        function getStatusText(status) {
            switch (status) {
                case 'checking': return 'In progress...';
                case 'success': return 'Completed successfully';
                case 'error': return 'Failed';
                default: return 'Waiting...';
            }
        }

        function updateProgress(percent) {
            document.getElementById('progress-fill').style.width = percent + '%';
        }

        function showSuccess() {
            document.getElementById('steps-container').innerHTML = `
                <div class="success-message">
                    <h2>🎉 Installation Completed Successfully!</h2>
                    <p>Your HireCameroon application is now ready to use.</p>

                    <div class="next-steps">
                        <h4>📝 Next Steps:</h4>
                        <ul>
                            <li><strong>Delete this install.php file</strong> for security</li>
                            <li><strong>Visit your website:</strong> <a href="https://hirecameroon.com" target="_blank">https://hirecameroon.com</a></li>
                            <li><strong>Test admin access:</strong> ${configData.admin_email || 'admin@hirecameroon.com'} / ${configData.admin_password || 'admin123'}</li>
                            <li><strong>Configure additional email settings</strong> if needed</li>
                            <li><strong>Set up payment integration</strong> (Mesomb) if not configured</li>
                        </ul>
                    </div>

                    <a href="https://hirecameroon.com" class="btn" style="margin-top: 20px;">🚀 Visit Your Website</a>
                </div>
            `;

            document.getElementById('controls').innerHTML = `
                <p style="color: #666; margin-top: 20px;">
                    <strong>⚠️ Security Notice:</strong> Please delete the install.php file after verifying everything works.
                </p>
            `;
        }
    </script>
</body>
</html>

<?php
// Backend processing for AJAX requests
if (isset($_GET['action'])) {
    header('Content-Type: application/json');

    try {
        require_once 'vendor/autoload.php';
        $app = require_once 'bootstrap/app.php';
        $kernel = $app->make('Illuminate\Contracts\Console\Kernel');
        $kernel->bootstrap();

        $action = $_GET['action'];

        switch ($action) {
            case 'loadLaravel':
                echo json_encode(['success' => true, 'message' => 'Laravel framework loaded successfully']);
                break;

            case 'checkDatabase':
                $app->make('db')->connection()->getPdo();
                echo json_encode(['success' => true, 'message' => 'Database connection successful']);
                break;

            case 'runMigrations':
                $exitCode = $kernel->call('migrate', ['--force' => true]);
                if ($exitCode === 0) {
                    echo json_encode(['success' => true, 'message' => 'Database migrations completed']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Migration failed', 'details' => $kernel->output()]);
                }
                break;

            case 'seedDatabase':
                $exitCode = $kernel->call('db:seed', ['--force' => true]);
                if ($exitCode === 0) {
                    echo json_encode(['success' => true, 'message' => 'Database seeded successfully']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Seeding failed', 'details' => $kernel->output()]);
                }
                break;

            case 'optimizeApp':
                $kernel->call('config:cache');
                $kernel->call('route:cache');
                $kernel->call('view:cache');
                echo json_encode(['success' => true, 'message' => 'Application optimized']);
                break;

            case 'finalize':
                if (empty(env('APP_KEY'))) {
                    $kernel->call('key:generate');
                }
                echo json_encode(['success' => true, 'message' => 'Installation finalized']);
                break;

            default:
                echo json_encode(['success' => false, 'message' => 'Unknown action']);
        }

    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Installation step failed',
            'details' => $e->getMessage()
        ]);
    }

    exit;
}
?>