<x-layouts.app>
    <x-slot name="header">
        <div class="flex items-center justify-between">
            <h2 class="font-semibold text-xl text-gray-800 leading-tight">
                User Activity Monitoring
            </h2>
            <div class="flex items-center space-x-4">
                <select id="timeRange" class="px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                    <option value="7">Last 7 days</option>
                    <option value="30" selected>Last 30 days</option>
                    <option value="90">Last 90 days</option>
                </select>
                <button id="exportBtn" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition">
                    Export Report
                </button>
            </div>
        </div>
    </x-slot>

    <!-- Activity Statistics -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <div class="bg-white rounded-2xl shadow-lg p-6 border border-gray-100">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-gray-600 text-sm font-medium">Total Activities</p>
                    <p class="text-3xl font-black text-black" id="totalActivities">0</p>
                    <p class="text-sm text-gray-500">last 30 days</p>
                </div>
                <div class="w-12 h-12 bg-gradient-to-br from-blue-500 to-blue-600 rounded-xl flex items-center justify-center">
                    <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"/>
                    </svg>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-2xl shadow-lg p-6 border border-gray-100">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-gray-600 text-sm font-medium">Active Users</p>
                    <p class="text-3xl font-black text-black" id="activeUsers">0</p>
                    <p class="text-sm text-gray-500">unique users</p>
                </div>
                <div class="w-12 h-12 bg-gradient-to-br from-green-500 to-green-600 rounded-xl flex items-center justify-center">
                    <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197m13.5-9a2.5 2.5 0 11-5 0 2.5 2.5 0 015 0z"/>
                    </svg>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-2xl shadow-lg p-6 border border-gray-100">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-gray-600 text-sm font-medium">Avg. Daily Activity</p>
                    <p class="text-3xl font-black text-black" id="avgDailyActivity">0</p>
                    <p class="text-sm text-gray-500">actions per day</p>
                </div>
                <div class="w-12 h-12 bg-gradient-to-br from-purple-500 to-purple-600 rounded-xl flex items-center justify-center">
                    <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6"/>
                    </svg>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-2xl shadow-lg p-6 border border-gray-100">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-gray-600 text-sm font-medium">Most Active User</p>
                    <p class="text-3xl font-black text-black" id="mostActiveUser">-</p>
                    <p class="text-sm text-gray-500" id="mostActiveCount">0 activities</p>
                </div>
                <div class="w-12 h-12 bg-gradient-to-br from-yellow-400 to-yellow-500 rounded-xl flex items-center justify-center">
                    <svg class="w-6 h-6 text-black" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 3v4M3 5h4M6 17v4m-2-2h4m5-16l2.286 6.857L21 12l-5.714 2.143L13 21l-2.286-6.857L5 12l5.714-2.143L13 3z"/>
                    </svg>
                </div>
            </div>
        </div>
    </div>

    <!-- Activity Charts -->
    <div class="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-8">
        <!-- Daily Activity Trend -->
        <div class="bg-white rounded-2xl shadow-lg p-6 border border-gray-100">
            <div class="flex items-center justify-between mb-6">
                <h3 class="text-lg font-bold text-gray-900">Daily Activity Trend</h3>
                <div class="flex items-center space-x-2">
                    <div class="w-3 h-3 bg-blue-500 rounded-full"></div>
                    <span class="text-sm text-gray-600">Activities</span>
                </div>
            </div>
            <canvas id="dailyActivityChart" class="w-full h-64"></canvas>
        </div>

        <!-- Top Actions -->
        <div class="bg-white rounded-2xl shadow-lg p-6 border border-gray-100">
            <div class="flex items-center justify-between mb-6">
                <h3 class="text-lg font-bold text-gray-900">Top Actions</h3>
                <div class="flex items-center space-x-2">
                    <div class="w-3 h-3 bg-green-500 rounded-full"></div>
                    <span class="text-sm text-gray-600">Frequency</span>
                </div>
            </div>
            <canvas id="topActionsChart" class="w-full h-64"></canvas>
        </div>
    </div>

    <!-- Recent Activity Feed -->
    <div class="bg-white rounded-2xl shadow-lg border border-gray-100">
        <div class="p-6 border-b border-gray-100">
            <h3 class="text-lg font-bold text-gray-900">Recent Activity Feed</h3>
            <p class="text-sm text-gray-600 mt-1">Latest user activities across the platform</p>
        </div>
        <div class="p-6">
            <div id="activityFeed" class="space-y-4">
                <!-- Activity items will be loaded here -->
            </div>
            <div class="mt-6 text-center">
                <button id="loadMoreBtn" class="bg-gray-100 hover:bg-gray-200 text-gray-700 px-4 py-2 rounded-lg text-sm font-medium transition">
                    Load More Activities
                </button>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            let currentPage = 1;
            const activitiesPerPage = 20;

            // Initialize charts
            let dailyActivityChart, topActionsChart;

            // Load initial data
            loadActivityData();
            loadActivityFeed();

            // Time range change handler
            document.getElementById('timeRange').addEventListener('change', function() {
                currentPage = 1;
                loadActivityData();
                loadActivityFeed();
            });

            // Load more button handler
            document.getElementById('loadMoreBtn').addEventListener('click', function() {
                currentPage++;
                loadActivityFeed();
            });

            // Export button handler
            document.getElementById('exportBtn').addEventListener('click', function() {
                const timeRange = document.getElementById('timeRange').value;
                window.location.href = `/admin/user-activity/export?days=${timeRange}`;
            });

            function loadActivityData() {
                const timeRange = document.getElementById('timeRange').value;

                fetch(`/admin/user-activity/data?days=${timeRange}`)
                    .then(response => response.json())
                    .then(data => {
                        updateStats(data.stats);
                        updateCharts(data);
                    })
                    .catch(error => console.error('Error loading activity data:', error));
            }

            function loadActivityFeed() {
                const timeRange = document.getElementById('timeRange').value;

                fetch(`/admin/user-activity/feed?days=${timeRange}&page=${currentPage}&per_page=${activitiesPerPage}`)
                    .then(response => response.json())
                    .then(data => {
                        if (currentPage === 1) {
                            document.getElementById('activityFeed').innerHTML = '';
                        }
                        renderActivityFeed(data.activities);

                        // Hide load more button if no more activities
                        if (!data.has_more) {
                            document.getElementById('loadMoreBtn').style.display = 'none';
                        }
                    })
                    .catch(error => console.error('Error loading activity feed:', error));
            }

            function updateStats(stats) {
                document.getElementById('totalActivities').textContent = stats.total_activities.toLocaleString();
                document.getElementById('activeUsers').textContent = stats.unique_users.toLocaleString();
                document.getElementById('avgDailyActivity').textContent = Math.round(stats.total_activities / parseInt(document.getElementById('timeRange').value));

                if (stats.most_active_user) {
                    document.getElementById('mostActiveUser').textContent = stats.most_active_user.name;
                    document.getElementById('mostActiveCount').textContent = `${stats.most_active_user.count} activities`;
                }
            }

            function updateCharts(data) {
                // Daily Activity Chart
                const dailyCtx = document.getElementById('dailyActivityChart').getContext('2d');
                if (dailyActivityChart) dailyActivityChart.destroy();

                dailyActivityChart = new Chart(dailyCtx, {
                    type: 'line',
                    data: {
                        labels: Object.keys(data.daily_activity),
                        datasets: [{
                            label: 'Activities',
                            data: Object.values(data.daily_activity),
                            borderColor: 'rgb(59, 130, 246)',
                            backgroundColor: 'rgba(59, 130, 246, 0.1)',
                            tension: 0.4,
                            fill: true
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: { legend: { display: false } },
                        scales: {
                            y: { beginAtZero: true, grid: { color: 'rgba(0, 0, 0, 0.1)' } },
                            x: { grid: { color: 'rgba(0, 0, 0, 0.1)' } }
                        }
                    }
                });

                // Top Actions Chart
                const actionsCtx = document.getElementById('topActionsChart').getContext('2d');
                if (topActionsChart) topActionsChart.destroy();

                topActionsChart = new Chart(actionsCtx, {
                    type: 'bar',
                    data: {
                        labels: data.top_actions.map(item => item.action.replace('_', ' ').toUpperCase()),
                        datasets: [{
                            label: 'Count',
                            data: data.top_actions.map(item => item.count),
                            backgroundColor: 'rgba(34, 197, 94, 0.8)',
                            borderColor: 'rgb(34, 197, 94)',
                            borderWidth: 1
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: { legend: { display: false } },
                        scales: {
                            y: { beginAtZero: true, grid: { color: 'rgba(0, 0, 0, 0.1)' } },
                            x: { grid: { color: 'rgba(0, 0, 0, 0.1)' } }
                        }
                    }
                });
            }

            function renderActivityFeed(activities) {
                const feedContainer = document.getElementById('activityFeed');

                activities.forEach(activity => {
                    const activityElement = createActivityElement(activity);
                    feedContainer.appendChild(activityElement);
                });
            }

            function createActivityElement(activity) {
                const div = document.createElement('div');
                div.className = 'flex items-start space-x-4 p-4 bg-gray-50 rounded-xl border border-gray-100';

                const iconDiv = document.createElement('div');
                iconDiv.className = 'w-10 h-10 bg-gradient-to-br from-blue-500 to-blue-600 rounded-lg flex items-center justify-center flex-shrink-0';
                iconDiv.innerHTML = getActivityIcon(activity.action);

                const contentDiv = document.createElement('div');
                contentDiv.className = 'flex-1 min-w-0';

                const headerDiv = document.createElement('div');
                headerDiv.className = 'flex items-center justify-between';

                const userSpan = document.createElement('span');
                userSpan.className = 'font-semibold text-gray-900';
                userSpan.textContent = activity.user.name;

                const timeSpan = document.createElement('span');
                timeSpan.className = 'text-sm text-gray-500';
                timeSpan.textContent = formatTimeAgo(activity.created_at);

                headerDiv.appendChild(userSpan);
                headerDiv.appendChild(timeSpan);

                const actionP = document.createElement('p');
                actionP.className = 'text-gray-700';
                actionP.textContent = getActivityDescription(activity);

                contentDiv.appendChild(headerDiv);
                contentDiv.appendChild(actionP);

                div.appendChild(iconDiv);
                div.appendChild(contentDiv);

                return div;
            }

            function getActivityIcon(action) {
                const icons = {
                    'login': '<svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 16l-4-4m0 0l4-4m-4 4h14m-5 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h7a3 3 0 013 3v1"/></svg>',
                    'logout': '<svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"/></svg>',
                    'view_job': '<svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 13.255A23.931 23.931 0 0112 15c-3.183 0-6.22-.62-9-1.745M16 6V4a2 2 0 00-2-2h-4a2 2 0 00-2 2v2m8 0V8a2 2 0 01-2 2H8a2 2 0 01-2-2V6m8 0H8"/></svg>',
                    'apply_job': '<svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/></svg>',
                    'post_job': '<svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"/></svg>',
                    'update_profile': '<svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/></svg>',
                    'view_company': '<svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4"/></svg>'
                };
                return icons[action] || '<svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/></svg>';
            }

            function getActivityDescription(activity) {
                const descriptions = {
                    'login': 'logged into the platform',
                    'logout': 'logged out of the platform',
                    'view_job': `viewed job "${activity.resource_name || 'Unknown Job'}"`,
                    'apply_job': `applied for job "${activity.resource_name || 'Unknown Job'}"`,
                    'post_job': `posted a new job "${activity.resource_name || 'Unknown Job'}"`,
                    'update_profile': 'updated their profile information',
                    'view_company': `viewed company "${activity.resource_name || 'Unknown Company'}"`
                };
                return descriptions[activity.action] || `${activity.action.replace('_', ' ')}`;
            }

            function formatTimeAgo(dateString) {
                const date = new Date(dateString);
                const now = new Date();
                const diffInSeconds = Math.floor((now - date) / 1000);

                if (diffInSeconds < 60) return 'just now';
                if (diffInSeconds < 3600) return `${Math.floor(diffInSeconds / 60)}m ago`;
                if (diffInSeconds < 86400) return `${Math.floor(diffInSeconds / 3600)}h ago`;
                return `${Math.floor(diffInSeconds / 86400)}d ago`;
            }
        });
    </script>
</x-layouts.app>