@extends('layouts.app')

@section('content')
<div class="max-w-4xl mx-auto p-6">
    <h1 class="text-2xl font-semibold mb-4">Create an Advertisement</h1>

    <form id="ad-form" method="POST" action="{{ route('ads.initiate-payment') }}" enctype="multipart/form-data" class="space-y-4">
        @csrf

        <div>
            <label class="block text-sm font-medium">Ad Plan</label>
            <select name="ad_plan_id" required class="mt-1 block w-full rounded-md border-gray-300">
                @foreach($plans as $plan)
                    <option value="{{ $plan->id }}">{{ $plan->name }} — {{ number_format($plan->price_cents/100, 0) }} XAF / {{ $plan->days }} days</option>
                @endforeach
            </select>
        </div>

        <div>
            <label class="block text-sm font-medium">Title</label>
            <input name="title" class="mt-1 block w-full rounded-md border-gray-300" placeholder="Short headline (optional)">
        </div>

        <div>
            <label class="block text-sm font-medium">Ad Type</label>
            <select name="type" id="ad-type" required class="mt-1 block w-full rounded-md border-gray-300">
                <option value="image">Image Ad</option>
                <option value="video">Video Ad</option>
                <option value="text">Text Ad</option>
            </select>
        </div>

        <div>
            <label class="block text-sm font-medium">Position</label>
            <select name="position" required class="mt-1 block w-full rounded-md border-gray-300">
                <option value="sidebar">Sidebar</option>
                <option value="banner">Top Banner</option>
                <option value="content">In-Content</option>
                <option value="footer">Footer</option>
            </select>
        </div>

        <div>
            <label class="block text-sm font-medium">Target URL</label>
            <input name="target_url" type="url" class="mt-1 block w-full rounded-md border-gray-300" placeholder="https://example.com">
        </div>

        <div>
            <label class="block text-sm font-medium">Start At (optional)</label>
            <input name="start_at" type="datetime-local" class="mt-1 block w-full rounded-md border-gray-300">
        </div>

        <div>
            <label class="block text-sm font-medium">Phone (for payment)</label>
            <input name="phone" type="text" required class="mt-1 block w-full rounded-md border-gray-300" placeholder="+2376...">
        </div>

        <div id="image-upload" class="ad-creative">
            <label class="block text-sm font-medium">Ad Image</label>
            <input id="image" name="image" type="file" accept="image/*" class="mt-1 block w-full">
            <p class="text-xs text-gray-500 mt-1">Upload your ad image here. Recommended size: 800x400px.</p>
        </div>

        <div id="video-upload" class="ad-creative hidden">
            <label class="block text-sm font-medium">Video URL</label>
            <input name="video_url" type="url" class="mt-1 block w-full rounded-md border-gray-300" placeholder="YouTube or Vimeo URL">
            <p class="text-xs text-gray-500 mt-1">Paste your YouTube or Vimeo video URL here.</p>

            <div class="mt-4">
                <label class="block text-sm font-medium">Video Thumbnail</label>
                <input id="poster" name="poster" type="file" accept="image/*" class="mt-1 block w-full">
                <p class="text-xs text-gray-500 mt-1">Upload a thumbnail image for your video. This will be shown before the video plays.</p>
            </div>
        </div>

        <div id="preview" class="mt-4 p-4 border rounded-lg bg-white hidden">
            <h3 class="font-semibold">Preview</h3>
            <div id="preview-image" class="mt-2"></div>
            <div id="preview-title" class="mt-2 font-medium text-lg"></div>
            <a id="preview-link" class="text-indigo-600 mt-1 block"></a>
        </div>

        <div class="flex items-center space-x-2">
            <button id="preview-btn" type="button" class="px-4 py-2 bg-gray-200 rounded">Preview</button>
            <button type="submit" class="px-4 py-2 bg-indigo-600 text-white rounded">Pay & Publish</button>
        </div>
    </form>
</div>

<script>
// Handle ad type switching
document.getElementById('ad-type').addEventListener('change', function() {
    const type = this.value;
    document.querySelectorAll('.ad-creative').forEach(el => el.classList.add('hidden'));
    
    if (type === 'image') {
        document.getElementById('image-upload').classList.remove('hidden');
    } else if (type === 'video') {
        document.getElementById('video-upload').classList.remove('hidden');
    }
});

// Preview functionality
document.getElementById('preview-btn').addEventListener('click', function () {
    const form = document.getElementById('ad-form');
    const type = form.querySelector('select[name="type"]').value;
    const title = form.querySelector('input[name="title"]').value;
    const url = form.querySelector('input[name="target_url"]').value;
    const position = form.querySelector('select[name="position"]').value;

    const preview = document.getElementById('preview');
    const previewImage = document.getElementById('preview-image');
    const previewTitle = document.getElementById('preview-title');
    const previewLink = document.getElementById('preview-link');

    previewTitle.textContent = title || 'Your Ad Title';
    previewLink.textContent = url || '';
    previewLink.href = url || '#';

    if (type === 'image') {
        const imgInput = document.getElementById('image');
        if (imgInput.files && imgInput.files[0]) {
            const reader = new FileReader();
            reader.onload = function (e) {
                previewImage.innerHTML = '<img src="'+e.target.result+'" alt="creative" style="max-width:320px; height:auto;">';
                preview.classList.remove('hidden');
            };
            reader.readAsDataURL(imgInput.files[0]);
        }
    } else if (type === 'video') {
        const videoUrl = form.querySelector('input[name="video_url"]').value;
        const posterInput = document.getElementById('poster');
        
        let videoHtml = '';
        if (videoUrl) {
            // Extract video ID and create embed
            const videoId = extractVideoId(videoUrl);
            if (videoId) {
                if (videoUrl.includes('youtube')) {
                    videoHtml = `<iframe width="320" height="180" src="https://www.youtube.com/embed/${videoId}" frameborder="0" allowfullscreen></iframe>`;
                } else if (videoUrl.includes('vimeo')) {
                    videoHtml = `<iframe width="320" height="180" src="https://player.vimeo.com/video/${videoId}" frameborder="0" allowfullscreen></iframe>`;
                }
            }
        }
        
        if (posterInput.files && posterInput.files[0]) {
            const reader = new FileReader();
            reader.onload = function (e) {
                previewImage.innerHTML = `
                    <div class="video-preview">
                        <img src="${e.target.result}" alt="video thumbnail" style="max-width:320px; height:auto;">
                        ${videoHtml}
                    </div>`;
                preview.classList.remove('hidden');
            };
            reader.readAsDataURL(posterInput.files[0]);
        } else {
            previewImage.innerHTML = videoHtml;
            preview.classList.remove('hidden');
        }
    } else {
        // Text-only ad
        previewImage.innerHTML = '';
        preview.classList.remove('hidden');
    }
});

// Helper function to extract video ID from URL
function extractVideoId(url) {
    let videoId = null;
    
    if (url.includes('youtube.com') || url.includes('youtu.be')) {
        if (url.includes('youtu.be/')) {
            videoId = url.split('youtu.be/')[1];
        } else {
            const urlParams = new URLSearchParams(new URL(url).search);
            videoId = urlParams.get('v');
        }
    } else if (url.includes('vimeo.com')) {
        videoId = url.split('vimeo.com/')[1];
    }
    
    return videoId;
}
</script>

@endsection
