<?php

use App\Http\Controllers\ProfileController;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\PaymentWebhookController;
use App\Http\Controllers\InstallController;
use App\Http\Controllers\CompanyController;
use App\Http\Controllers\SubscribeController;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
|
| Here is where you can register web routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "web" middleware group. Make something great!
|
*/

Route::get('/', function () {
    $featuredJobsQuery = \App\Models\Job::with(['organization'])
        ->where('status', 'active');

    // Only apply application_deadline filter if column exists to avoid SQL errors during migrations
    if (\Illuminate\Support\Facades\Schema::hasColumn('jobs_optimized', 'application_deadline')) {
        $featuredJobsQuery->where(function ($q) {
            $q->where('application_deadline', '>', now())
              ->orWhereNull('application_deadline');
        });
    }

    $featuredJobs = $featuredJobsQuery
        ->orderBy('created_at', 'desc')
        ->limit(8)
        ->get();

    return view('welcome', compact('featuredJobs'));
})->name('welcome');

// Mesomb payment webhook
Route::post('/webhooks/mesomb', [PaymentWebhookController::class, 'handle'])->name('webhooks.mesomb');

// Installer (accessible only before installation)
Route::middleware([])->group(function () {
    Route::get('/install', [InstallController::class, 'index'])->name('install.index');
    Route::post('/install', [InstallController::class, 'store'])->name('install.store');
    Route::get('/install/success', [InstallController::class, 'success'])->name('install.success');
});

// Marketing pages
Route::get('/pricing', function () {
    return view('marketing.pricing');
})->name('pricing');

Route::get('/about', function () {
    return view('marketing.about');
})->name('about');

Route::get('/contact', function () {
    return view('marketing.contact');
})->name('contact');

Route::get('/faq', function () {
    return view('marketing.faq');
})->name('faq');

Route::get('/news', function () {
    return view('marketing.news');
})->name('news');

Route::get('/privacy', function () {
    return view('marketing.privacy');
})->name('privacy');

Route::view('/donate', 'donate')->name('donate');

// Job pages
Route::get('/jobs', [App\Http\Controllers\JobController::class, 'index'])->name('jobs.index');
Route::get('/jobs/{job}', [App\Http\Controllers\JobController::class, 'show'])->name('jobs.show');

// Job creation
Route::get('/jobs/create', [App\Http\Controllers\JobController::class, 'create'])->name('jobs.create');
Route::post('/jobs', [App\Http\Controllers\JobController::class, 'store'])->name('jobs.store');

// Job creation (redirect to dashboard for authenticated users with company)
Route::middleware(['auth', 'verified'])->group(function () {
    // Ad payment initiation (advertisers)
    Route::post('/ads/initiate-payment', [App\Http\Controllers\AdPaymentController::class, 'initiate'])->name('ads.initiate-payment');
    // Ad creation UI
    Route::get('/ads/create', [App\Http\Controllers\AdController::class, 'create'])->name('ads.create');
    // Advertiser dashboard - list and show own ads
    Route::get('/ads', [App\Http\Controllers\AdController::class, 'index'])->name('ads.index');
    Route::get('/ads/{ad}', [App\Http\Controllers\AdController::class, 'show'])->name('ads.show');
    // Ad tracking endpoints
    Route::post('/ads/{ad}/impression', [App\Http\Controllers\AdController::class, 'recordImpression'])->name('ads.impression');
    Route::post('/ads/{ad}/click', [App\Http\Controllers\AdController::class, 'recordClick'])->name('ads.click');
});

// Company pages
Route::get('/companies', [CompanyController::class, 'index'])->name('companies.index');
Route::get('/companies/{company}', [CompanyController::class, 'show'])->name('companies.show');
Route::get('/companies/{company}/manage', [CompanyController::class, 'showCompany'])->name('companies.manage');

// Reviews
Route::post('/reviews', [App\Http\Controllers\ReviewController::class, 'store'])->name('reviews.store');
Route::get('/reviews', [App\Http\Controllers\ReviewController::class, 'index'])->name('reviews.index');

// Application pages
Route::middleware(['auth', 'verified'])->group(function () {
    Route::get('/applications', [App\Http\Controllers\ApplicationController::class, 'index'])->name('applications.index');
    Route::get('/applications/create/{job}', [App\Http\Controllers\ApplicationController::class, 'create'])->name('applications.create');
    Route::post('/applications', [App\Http\Controllers\ApplicationController::class, 'store'])->name('applications.store');
    Route::get('/applications/{application}', [App\Http\Controllers\ApplicationController::class, 'show'])->name('applications.show');
    Route::patch('/applications/{application}/withdraw', [App\Http\Controllers\ApplicationController::class, 'withdraw'])->name('applications.withdraw');
    Route::get('/applications/{application}/resume/download', [App\Http\Controllers\ApplicationController::class, 'downloadResume'])->name('applications.download-resume');
});

// Company onboarding
Route::middleware(['auth', 'verified'])->group(function () {
    Route::get('/companies/create', [CompanyController::class, 'create'])->name('companies.create');
    Route::post('/companies', [CompanyController::class, 'store'])->name('companies.store');
    Route::middleware(['company.owner'])->group(function () {
        Route::get('/companies/{company}/subscribe', [SubscribeController::class, 'show'])->name('subscriptions.subscribe');
        Route::post('/companies/{company}/subscribe', [SubscribeController::class, 'start'])->name('subscriptions.start');
    });
});

// Company Dashboard Routes
Route::middleware(['auth', 'verified', 'company.owner'])->prefix('dashboard')->group(function () {
    Route::get('/', [App\Http\Controllers\CompanyDashboardController::class, 'index'])->name('dashboard.company');
    Route::get('/jobs', [App\Http\Controllers\CompanyDashboardController::class, 'jobs'])->name('dashboard.jobs');
    Route::get('/jobs/create', [App\Http\Controllers\CompanyDashboardController::class, 'createJob'])->name('dashboard.jobs.create');
    Route::post('/jobs', [App\Http\Controllers\CompanyDashboardController::class, 'storeJob'])->name('dashboard.jobs.store');
    Route::get('/applications', [App\Http\Controllers\CompanyDashboardController::class, 'applications'])->name('dashboard.applications');
    Route::get('/analytics', [App\Http\Controllers\CompanyDashboardController::class, 'analytics'])->name('dashboard.analytics');
});

// Admin Routes - Accessible at /admin
Route::prefix('admin')->group(function () {
    // Admin Login (if not authenticated)
    Route::get('/login', function () {
        if (auth()->check() && auth()->user()->is_admin) {
            return redirect('/admin');
        }
        return view('admin.login');
    })->name('admin.login');

    // Admin Authentication
    Route::post('/login', [App\Http\Controllers\AdminController::class, 'authenticate'])->name('admin.authenticate');
    Route::post('/logout', [App\Http\Controllers\AdminController::class, 'logout'])->name('admin.logout');

    // Protected Admin Routes
    Route::middleware(['auth', 'verified', 'admin'])->group(function () {
        Route::get('/', [App\Http\Controllers\AdminController::class, 'dashboard'])->name('admin.dashboard');
        Route::get('/users', [App\Http\Controllers\AdminController::class, 'users'])->name('admin.users');
        Route::get('/jobs', [App\Http\Controllers\AdminController::class, 'jobs'])->name('admin.jobs');
        Route::get('/applications', [App\Http\Controllers\AdminController::class, 'applications'])->name('admin.applications');
        Route::get('/analytics', [App\Http\Controllers\AdminController::class, 'analytics'])->name('admin.analytics');
        Route::get('/user-activity', [App\Http\Controllers\AdminController::class, 'userActivity'])->name('admin.user-activity');
        Route::get('/user-activity/data', [App\Http\Controllers\AdminController::class, 'getUserActivityData'])->name('admin.user-activity.data');
        Route::get('/user-activity/feed', [App\Http\Controllers\AdminController::class, 'getUserActivityFeed'])->name('admin.user-activity.feed');
        Route::get('/system-health', [App\Http\Controllers\AdminController::class, 'systemHealth'])->name('admin.system-health');
        Route::get('/system-health/data', [App\Http\Controllers\AdminController::class, 'getSystemHealthData'])->name('admin.system-health.data');
        Route::get('/audit-logs', [App\Http\Controllers\AdminController::class, 'auditLogs'])->name('admin.audit-logs');
        Route::get('/audit-logs/data', [App\Http\Controllers\AdminController::class, 'getAuditLogsData'])->name('admin.audit-logs.data');
        Route::get('/audit-logs/{id}', [App\Http\Controllers\AdminController::class, 'getAuditLogDetails'])->name('admin.audit-logs.details');
        Route::get('/audit-logs/export', [App\Http\Controllers\AdminController::class, 'exportAuditLogs'])->name('admin.audit-logs.export');
        Route::get('/bulk-email', [App\Http\Controllers\AdminController::class, 'bulkEmail'])->name('admin.bulk-email');
        Route::get('/bulk-email/data', [App\Http\Controllers\AdminController::class, 'getBulkEmailData'])->name('admin.bulk-email.data');
        Route::post('/bulk-email/campaigns', [App\Http\Controllers\AdminController::class, 'createEmailCampaign'])->name('admin.bulk-email.campaigns.store');
        Route::delete('/bulk-email/campaigns/{campaignId}', [App\Http\Controllers\AdminController::class, 'deleteEmailCampaign'])->name('admin.bulk-email.campaigns.delete');
        Route::get('/content-moderation', [App\Http\Controllers\AdminController::class, 'contentModeration'])->name('admin.content-moderation');
        Route::get('/content-moderation/data', [App\Http\Controllers\AdminController::class, 'getContentModerationData'])->name('admin.content-moderation.data');
        Route::get('/content-moderation/{type}/{id}', [App\Http\Controllers\AdminController::class, 'getContentDetails'])->name('admin.content-moderation.details');
        Route::post('/content-moderation/{type}/{id}/{action}', [App\Http\Controllers\AdminController::class, 'moderateContent'])->name('admin.content-moderation.moderate');
        Route::get('/settings', [App\Http\Controllers\AdminController::class, 'settings'])->name('admin.settings');
        Route::get('/backups', [App\Http\Controllers\AdminController::class, 'listBackups'])->name('admin.backups');
        Route::post('/backups/create', [App\Http\Controllers\AdminController::class, 'createBackup'])->name('admin.backup.create');
        Route::get('/backups/{backupName}/download', [App\Http\Controllers\AdminController::class, 'downloadBackup'])->name('admin.backup.download');
        Route::delete('/backups/{backupName}', [App\Http\Controllers\AdminController::class, 'deleteBackup'])->name('admin.backup.delete');




        // SEO Management
        Route::get('/seo-management', [App\Http\Controllers\AdminController::class, 'seoManagement'])->name('admin.seo-management');
        Route::post('/seo-management/meta-tags', [App\Http\Controllers\AdminController::class, 'updateMetaTags'])->name('admin.seo-management.meta-tags');
        Route::post('/seo-management/keywords', [App\Http\Controllers\AdminController::class, 'addKeyword'])->name('admin.seo-management.keywords');
        Route::put('/seo-management/keywords/{keyword}', [App\Http\Controllers\AdminController::class, 'updateKeyword'])->name('admin.seo-management.keywords.update');
        Route::post('/seo-management/keyword-research', [App\Http\Controllers\AdminController::class, 'runKeywordResearch'])->name('admin.seo-management.keyword-research');
        Route::post('/seo-management/generate-sitemap', [App\Http\Controllers\AdminController::class, 'generateSitemap'])->name('admin.seo-management.generate-sitemap');
        Route::post('/seo-management/submit-sitemap', [App\Http\Controllers\AdminController::class, 'submitSitemap'])->name('admin.seo-management.submit-sitemap');
        Route::get('/seo-management/analytics', [App\Http\Controllers\AdminController::class, 'getSEOAnalytics'])->name('admin.seo-management.analytics');

        // Integration Management
        Route::get('/integration-management', [App\Http\Controllers\AdminController::class, 'integrationManagement'])->name('admin.integration-management');
        Route::get('/integration-management/api-keys', [App\Http\Controllers\AdminController::class, 'getApiKeys'])->name('admin.integration-management.api-keys');
        Route::post('/integration-management/api-keys', [App\Http\Controllers\AdminController::class, 'createApiKey'])->name('admin.integration-management.api-keys.create');
        Route::post('/integration-management/api-keys/{keyId}/toggle', [App\Http\Controllers\AdminController::class, 'toggleApiKey'])->name('admin.integration-management.api-keys.toggle');
        Route::delete('/integration-management/api-keys/{keyId}', [App\Http\Controllers\AdminController::class, 'deleteApiKey'])->name('admin.integration-management.api-keys.delete');
        Route::get('/integration-management/webhooks', [App\Http\Controllers\AdminController::class, 'getWebhooks'])->name('admin.integration-management.webhooks');
        Route::get('/integration-management/logs', [App\Http\Controllers\AdminController::class, 'getIntegrationLogs'])->name('admin.integration-management.logs');
        Route::post('/integration-management/test-payment', [App\Http\Controllers\AdminController::class, 'testPaymentIntegration'])->name('admin.integration-management.test-payment');
        Route::get('/integration-management/logs/export', [App\Http\Controllers\AdminController::class, 'exportIntegrationLogs'])->name('admin.integration-management.logs.export');

        // User Management Routes
        Route::resource('users', App\Http\Controllers\Admin\UserController::class, ['as' => 'admin']);
        Route::patch('/users/{user}/status', [App\Http\Controllers\Admin\UserController::class, 'toggleStatus'])->name('admin.users.toggle-status');
        Route::patch('/users/{user}/reset-password', [App\Http\Controllers\Admin\UserController::class, 'resetPassword'])->name('admin.users.reset-password');
        Route::post('/users/{user}/impersonate', [App\Http\Controllers\Admin\UserController::class, 'impersonate'])->name('admin.users.impersonate');
        Route::post('/users/stop-impersonating', [App\Http\Controllers\Admin\UserController::class, 'stopImpersonating'])->name('admin.users.stop-impersonating');

        // Job Management Routes
        Route::resource('jobs', App\Http\Controllers\Admin\JobController::class, ['as' => 'admin']);
        Route::patch('/jobs/{job}/status', [App\Http\Controllers\Admin\JobController::class, 'updateStatus'])->name('admin.jobs.update-status');
        Route::patch('/jobs/{job}/verification', [App\Http\Controllers\Admin\JobController::class, 'updateVerification'])->name('admin.jobs.update-verification');
        Route::patch('/jobs/{job}/featured', [App\Http\Controllers\Admin\JobController::class, 'toggleFeatured'])->name('admin.jobs.toggle-featured');
        Route::post('/jobs/{job}/flag-scam', [App\Http\Controllers\Admin\JobController::class, 'flagAsScam'])->name('admin.jobs.flag-scam');
        Route::delete('/jobs/{job}/remove-scam-flag', [App\Http\Controllers\Admin\JobController::class, 'removeScamFlag'])->name('admin.jobs.remove-scam-flag');
        Route::post('/jobs/bulk-action', [App\Http\Controllers\Admin\JobController::class, 'bulkAction'])->name('admin.jobs.bulk-action');

        // Review Management Routes
        Route::resource('reviews', App\Http\Controllers\Admin\ReviewController::class, ['as' => 'admin']);
        Route::patch('/reviews/{review}/featured', [App\Http\Controllers\Admin\ReviewController::class, 'toggleFeatured'])->name('admin.reviews.toggle-featured');
        Route::patch('/reviews/{review}/published', [App\Http\Controllers\Admin\ReviewController::class, 'togglePublished'])->name('admin.reviews.toggle-published');

        // News Management Routes
        Route::resource('news', App\Http\Controllers\Admin\NewsController::class, ['as' => 'admin']);
        Route::patch('/news/{news}/featured', [App\Http\Controllers\Admin\NewsController::class, 'toggleFeatured'])->name('admin.news.toggle-featured');
        Route::patch('/news/{news}/published', [App\Http\Controllers\Admin\NewsController::class, 'togglePublished'])->name('admin.news.toggle-published');

        // Featured Content Management Routes
        Route::resource('featured-content', App\Http\Controllers\Admin\FeaturedContentController::class, ['as' => 'admin']);
        Route::patch('/featured-content/{featuredContent}/active', [App\Http\Controllers\Admin\FeaturedContentController::class, 'toggleActive'])->name('admin.featured-content.toggle-active');
        Route::post('/featured-content/update-order', [App\Http\Controllers\Admin\FeaturedContentController::class, 'updateOrder'])->name('admin.featured-content.update-order');

        // Email Template Management Routes
        Route::resource('email-templates', App\Http\Controllers\Admin\EmailTemplateController::class, ['as' => 'admin']);
        Route::patch('/email-templates/{emailTemplate}/active', [App\Http\Controllers\Admin\EmailTemplateController::class, 'toggleActive'])->name('admin.email-templates.toggle-active');
        Route::post('/email-templates/{emailTemplate}/preview', [App\Http\Controllers\Admin\EmailTemplateController::class, 'preview'])->name('admin.email-templates.preview');
        Route::post('/email-templates/{emailTemplate}/send-test', [App\Http\Controllers\Admin\EmailTemplateController::class, 'sendTest'])->name('admin.email-templates.send-test');

        // System Update Management Routes
        Route::resource('system-updates', App\Http\Controllers\Admin\SystemUpdateController::class, ['as' => 'admin']);
        Route::patch('/system-updates/{systemUpdate}/published', [App\Http\Controllers\Admin\SystemUpdateController::class, 'togglePublished'])->name('admin.system-updates.toggle-published');
        Route::get('/system-updates/published', [App\Http\Controllers\Admin\SystemUpdateController::class, 'published'])->name('admin.system-updates.published');

        // Ad Management Routes
        Route::resource('ads', App\Http\Controllers\Admin\AdController::class, ['as' => 'admin']);
        Route::patch('/ads/{ad}/toggle-status', [App\Http\Controllers\Admin\AdController::class, 'toggleStatus'])->name('admin.ads.toggle-status');

        // Legacy Admin Actions (keeping for backward compatibility)
        Route::patch('/users/{user}/status', [App\Http\Controllers\AdminController::class, 'updateUserStatus'])->name('admin.users.update-status');
        Route::patch('/jobs/{job}/status', [App\Http\Controllers\AdminController::class, 'updateJobStatus'])->name('admin.jobs.update-status');
        Route::delete('/jobs/{job}', [App\Http\Controllers\AdminController::class, 'deleteJob'])->name('admin.jobs.delete');
        Route::delete('/users/{user}', [App\Http\Controllers\AdminController::class, 'deleteUser'])->name('admin.users.delete');
        Route::patch('/settings', [App\Http\Controllers\AdminController::class, 'updateSettings'])->name('admin.settings.update');
    });
});

// Legacy dashboard route - redirect to new dashboard
Route::get('/dashboard', function () {
    return redirect('/dashboard');
})->middleware(['auth', 'verified'])->name('dashboard');

Route::middleware('auth')->group(function () {
    Route::get('/profile', [App\Http\Controllers\ProfileController::class, 'edit'])->name('profile.edit');
    Route::patch('/profile', [App\Http\Controllers\ProfileController::class, 'update'])->name('profile.update');
    Route::patch('/profile/professional', [App\Http\Controllers\ProfileController::class, 'updateProfessional'])->name('profile.update-professional');
    Route::patch('/profile/company', [App\Http\Controllers\ProfileController::class, 'updateCompany'])->name('profile.update-company');
    Route::patch('/profile/preferences', [App\Http\Controllers\ProfileController::class, 'updatePreferences'])->name('profile.update-preferences');
    Route::put('/profile/password', [App\Http\Controllers\ProfileController::class, 'updatePassword'])->name('profile.password.update');
    Route::get('/profile/resume/download', [App\Http\Controllers\ProfileController::class, 'downloadResume'])->name('profile.download-resume');
    Route::delete('/profile', [App\Http\Controllers\ProfileController::class, 'destroy'])->name('profile.destroy');
});

require __DIR__.'/auth.php';
