#!/bin/bash

# HireCameroon Git Setup Automation Script
# This script will automatically set up Git with SSH keys and push your first commit

echo "🚀 HireCameroon Git Setup Automation"
echo "===================================="

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Function to print colored output
print_status() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# Check if we're in the right directory
if [ ! -f "composer.json" ] || [ ! -f "artisan" ]; then
    print_error "Please run this script from the HireCameroon project root directory"
    exit 1
fi

print_status "Setting up Git for HireCameroon project..."

# Check if Git is installed
if ! command -v git &> /dev/null; then
    print_error "Git is not installed. Please install Git first."
    exit 1
fi

# Check if SSH key already exists
SSH_KEY_PATH="$HOME/.ssh/id_hirecameroon"
if [ -f "$SSH_KEY_PATH" ]; then
    print_warning "SSH key already exists at $SSH_KEY_PATH"
    read -p "Do you want to overwrite it? (y/N): " -n 1 -r
    echo
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        print_status "Using existing SSH key"
    else
        print_status "Generating new SSH key..."
        rm -f "$SSH_KEY_PATH" "$SSH_KEY_PATH.pub"
        ssh-keygen -t dsa -b 1024 -f "$SSH_KEY_PATH" -N "" -C "hirecameroon-git-key"
    fi
else
    print_status "Generating SSH key for HireCameroon..."
    ssh-keygen -t dsa -b 1024 -f "$SSH_KEY_PATH" -N "" -C "hirecameroon-git-key"
fi

# Start SSH agent and add key
print_status "Starting SSH agent and adding key..."
eval "$(ssh-agent -s)" > /dev/null 2>&1
ssh-add "$SSH_KEY_PATH" > /dev/null 2>&1

# Display public key
print_success "SSH Key generated successfully!"
echo
echo "=================================================================================="
echo "COPY THIS PUBLIC KEY TO GITHUB:"
echo "=================================================================================="
cat "$SSH_KEY_PATH.pub"
echo
echo "=================================================================================="
echo
echo "Instructions:"
echo "1. Go to https://github.com"
echo "2. Sign in to your account"
echo "3. Click your profile picture → Settings"
echo "4. Click 'SSH and GPG keys' in the left sidebar"
echo "5. Click 'New SSH key'"
echo "6. Title: HireCameroon Repository Key"
echo "7. Paste the key above into the 'Key' field"
echo "8. Click 'Add SSH key'"
echo
read -p "Press Enter after you've added the SSH key to GitHub..."

# Configure Git
print_status "Configuring Git..."
git config --global user.name "Aloah Milton"
git config --global user.email "aloahmilton@gmail.com"
git config --global core.sshCommand "ssh -i $SSH_KEY_PATH -F /dev/null"

# Initialize Git repository if not already done
if [ ! -d ".git" ]; then
    print_status "Initializing Git repository..."
    git init
fi

# Check if remote origin already exists
if git remote get-url origin > /dev/null 2>&1; then
    print_warning "Git remote 'origin' already exists"
    git remote get-url origin
    read -p "Do you want to change it? (y/N): " -n 1 -r
    echo
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        read -p "Enter your GitHub repository URL (e.g., git@github.com:username/hirecameroon.git): " repo_url
        git remote set-url origin "$repo_url"
    fi
else
    read -p "Enter your GitHub repository URL (e.g., git@github.com:username/hirecameroon.git): " repo_url
    git remote add origin "$repo_url"
fi

# Add all files
print_status "Adding all files to Git..."
git add .

# Create the epic first commit
print_status "Creating your first commit..."
git commit -m "🚀 Initial commit: HireCameroon Job Portal Platform

✨ Complete job portal platform for Cameroon
📊 12 optimized database tables (66% reduction from 35+)
🎨 Laravel 9.19 + Alpine.js + Tailwind CSS + PHPMailer
🔐 JWT authentication, role-based permissions
💰 Mesomb mobile money integration ready
📱 Responsive design with professional UI
📚 Interactive HTML documentation website
🐳 Docker environment configured
🤖 Automated FTP deployment with GitHub Actions
🇨🇲 Built for the Cameroonian job market

Features:
• Advanced job search & filtering
• Company profiles with verification
• One-click applications with resume upload
• Admin panel with content moderation
• Real-time notifications & analytics
• Mobile-optimized interface

Production ready & fully configured!"

# Test SSH connection to GitHub
print_status "Testing SSH connection to GitHub..."
if ssh -T git@github.com -o StrictHostKeyChecking=no > /dev/null 2>&1; then
    print_success "SSH connection to GitHub successful!"
else
    print_warning "SSH connection test failed. This might be normal if you haven't added the key yet."
    print_status "Continuing with push attempt..."
fi

# Push to GitHub
print_status "Pushing to GitHub..."
if git push -u origin main; then
    print_success "Successfully pushed to GitHub!"
    echo
    print_success "🎉 Congratulations! Your HireCameroon project is now live on GitHub!"
    echo
    echo "📊 Project Statistics:"
    echo "• 12 optimized database tables"
    echo "• 35+ Laravel migrations"
    echo "• Complete authentication system"
    echo "• Professional documentation"
    echo "• Docker deployment ready"
    echo "• Automated CI/CD pipeline"
    echo
    echo "🚀 Next Steps:"
    echo "1. Visit your GitHub repository"
    echo "2. Set up GitHub Actions secrets for deployment"
    echo "3. Configure domain and hosting"
    echo "4. Launch HireCameroon.com!"
    echo
    print_success "Your journey to revolutionize job searching in Cameroon begins now! 🇨🇲"
else
    print_error "Failed to push to GitHub. Please check:"
    echo "1. SSH key is added to GitHub"
    echo "2. Repository URL is correct"
    echo "3. You have write access to the repository"
    echo
    print_status "You can try pushing manually with: git push -u origin main"
    exit 1
fi

print_success "Git setup completed successfully!"