<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Company;
use App\Models\Job;
use App\Models\Application;
use Tests\TestCase;

class ApiTest extends TestCase
{
    protected $user;
    protected $company;
    protected $job;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create test user
        $this->user = User::factory()->create([
            'email' => 'test@hirecameroon.com',
        ]);

        // Create test company
        $this->company = Company::factory()->create([
            'owner_user_id' => $this->user->id,
            'name' => 'Test Company Ltd',
        ]);

        // Create test job
        $this->job = Job::factory()->create([
            'company_id' => $this->company->id,
            'title' => 'Senior Laravel Developer',
            'status' => 'active',
        ]);
    }

    /** @test */
    public function can_list_companies_with_search()
    {
        $response = $this->getJson('/api/companies?search=Test');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'success',
                'data' => [
                    'data' => [
                        '*' => [
                            'id',
                            'name',
                            'description',
                            'sector',
                            'created_at',
                        ]
                    ],
                    'current_page',
                    'total',
                ],
                'message',
            ]);
    }

    /** @test */
    public function can_create_company_profile()
    {
        $this->actingAs($this->user);

        $response = $this->postJson('/api/companies', [
            'name' => 'New Test Company',
            'description' => 'A test company description',
            'sector' => 'Technology',
            'location' => 'Douala, Cameroon',
        ]);

        $response->assertStatus(201)
            ->assertJsonStructure([
                'success',
                'data' => [
                    'id',
                    'name',
                    'description',
                    'sector',
                    'owner_user_id',
                ],
                'message',
            ]);

        $this->assertEquals('New Test Company', $response->json('data.name'));
    }

    /** @test */
    public function user_can_only_update_their_own_company()
    {
        $otherUser = User::factory()->create();
        $otherCompany = Company::factory()->create([
            'owner_user_id' => $otherUser->id,
        ]);

        $this->actingAs($this->user);

        $response = $this->putJson("/api/companies/{$otherCompany->id}", [
            'name' => 'Updated Company',
        ]);

        $response->assertStatus(403)
            ->assertJson(['success' => false]);
    }

    /** @test */
    public function can_get_user_profile()
    {
        $this->actingAs($this->user);

        $response = $this->getJson('/api/users/profile');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'success',
                'data' => [
                    'id',
                    'name',
                    'email',
                    'profile' => [
                        'phone',
                        'location',
                        'bio',
                        'current_job_title',
                    ],
                    'company' => [
                        'id',
                        'name',
                    ],
                ],
                'message',
            ]);
    }

    /** @test */
    public function can_update_user_profile()
    {
        $this->actingAs($this->user);

        $response = $this->putJson('/api/users/profile', [
            'first_name' => 'John',
            'last_name' => 'Doe',
            'email' => 'john.doe@test.com',
            'phone' => '+237123456789',
            'location' => 'Yaoundé, Cameroon',
            'bio' => 'Experienced Laravel developer',
            'current_job_title' => 'Software Engineer',
        ]);

        $response->assertStatus(200)
            ->assertJsonStructure([
                'success',
                'data' => [
                    'name',
                    'email',
                    'profile' => [
                        'phone',
                        'location',
                        'bio',
                    ],
                ],
                'message',
            ]);
    }

    /** @test */
    public function user_can_apply_to_job()
    {
        $this->actingAs($this->user);

        $response = $this->postJson("/api/jobs/{$this->job->id}/apply", [
            'cover_letter' => 'I am very interested in this position.',
            'expected_salary' => 800000,
            'availability_date' => '2024-02-01',
            'additional_info' => 'Available immediately.',
        ]);

        $response->assertStatus(201)
            ->assertJsonStructure([
                'success',
                'data' => [
                    'id',
                    'job_id',
                    'user_id',
                    'status',
                    'applied_at',
                ],
                'message',
            ]);

        $this->assertEquals('pending', $response->json('data.status'));
    }

    /** @test */
    public function user_cannot_apply_twice_to_same_job()
    {
        // First application
        $this->actingAs($this->user);
        $this->postJson("/api/jobs/{$this->job->id}/apply", [
            'cover_letter' => 'First application',
        ]);

        // Second application (should fail)
        $response = $this->postJson("/api/jobs/{$this->job->id}/apply", [
            'cover_letter' => 'Second application',
        ]);

        $response->assertStatus(400);
    }

    /** @test */
    public function api_requests_are_rate_limited()
    {
        // Make 65 requests to exceed the default limit of 60
        for ($i = 0; $i < 65; $i++) {
            $response = $this->getJson('/api/companies');
            if ($response->status() === 429) {
                break;
            }
        }

        $response->assertStatus(429)
            ->assertJson([
                'error' => 'Rate limit exceeded',
            ]);
    }
}